#!/usr/bin/env python
'''
Gain-Curve of IRAM 30-m. The result gives the real gain 
w.r.t. to the standard Gain (i.e. a factor).

Input parameters: [elevation in deg] [frequency in GHz]

Example:
> ./PVgaincurve.py 15 228
0.689586652005

means the telescope's gain is a ~31% lower at elev=15deg (f=228GHz) than the standard assumptions.

Pablo Torne, March 2015
'''

import numpy as np

class gainCalculator:
    """
    This class allows to calculate the normalized gain depending on frequency and elevation.
    It follows Panalver's report of 16-April-2012.
    
    Usage:
    - create an object:  gc = gainCalculator()
    - use the method G of the object, e.g. gc.G(elev_in_deg, freq_in_GHz)
    """
    def Elmax(self, f):
        return (1.567e-6*(f**3)) - (1.233e-3*(f**2)) + (3.194e-1*f) + (2.203e1)

    def Aeff0(self, elev):
        return ( (8.8466e-6*(elev**2)) - (1.2523e-3*elev) + (6.9608e-1) )

    def rms(self, elev):
        return ( (2.5523e-2*(elev**2)) - (2.5534*elev) + (1.1937e2)  )

    def Aeff(self, elev, lamb):
        return ( self.Aeff0(elev)*(np.e**( -1*(  ( (4*np.pi*self.rms(elev))/(lamb/1e-6) )**2 )  ) ) )

    def freq_to_lambda(self, freq):
        return ( 299792458./(freq*1e9) )

    def G(self, elev, freq):
        '''
        Calculate the gain correction factor at a given elevation and frequency.

        Usage: G(elev_in_deg, freq_in_GHz)
        '''
        lamb=self.freq_to_lambda(freq)
        #print "Frequency = %s GHz"%freq
        #print "Wavelength = %10.7f mm"%(lamb/1e-3)
        G = self.Aeff(elev,lamb) / self.Aeff(self.Elmax(freq),lamb)
        #print "G(%s deg) at %s GHz = %s"%(elev,freq,G)
        return G

if __name__ == '__main__':

    import sys

    # Check that input parameters are there
    if len(sys.argv) != 3:
        print "Please provide arguments: > PVgaincurve.py elevation(deg) frequency(GHz)"
        sys.exit()

    elev = float(sys.argv[1]) # Telescope elevation in degrees
    freq = float(sys.argv[2]) # Observing (central) frequency
    
    gainCalc = gainCalculator()
    print gainCalc.G(elev,freq)
