#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalFEDelays.py 239891 2017-01-16 20:20:03Z nphillip $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global math
import math

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global PyDataModelEnumeration
import PyDataModelEnumeration

global Observation
import Observation.AtmCalTarget
import Observation.OnOffTarget
import Observation.SSRTuning
import Observation.ObsCalBase

# Band6( ReceiverBand.ALMA_RB_06,211.0,275.0, SB.SB2, 5, 10,  6,   3, 73.6, 88.4),

class ObsCalSpectralCheckout(Observation.ObsCalBase.ObsCalBase):

    lo1List = []
    lo2List = []

    bandLOrange = {
        1:  [ 31.0,  38.5],
        2:  [ 79.0, 104.0],
        3:  [ 92.0, 108.0],
        4:  [133.0, 155.0],
        5:  [165.5, 203.0],
        6:  [221.0, 265.0],
        7:  [283.0, 365.0],
        8:  [393.0, 492.0],
        9:  [614.0, 708.0],
        10: [799.0, 938.0]
    }

    options = [
        Observation.ObsCalBase.scriptOption("band", int, 4),
        Observation.ObsCalBase.scriptOption("RepeatCount", int, 1),
        Observation.ObsCalBase.scriptOption("polarization", str, "2"),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 9.216),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("atmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("onOffSubscanDuration", float, 18.432),
        Observation.ObsCalBase.scriptOption("refAzOffset", float, 600.0),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("bbNames", str, ""),
        Observation.ObsCalBase.scriptOption("ambientLoad", bool, False),
        Observation.ObsCalBase.scriptOption("LO1", str, ""),
        Observation.ObsCalBase.scriptOption("LO2", str, ""),
        # Observation.ObsCalBase.scriptOption("mixerMode", str, ""),
    ]

    def parseOptions(self):
        self.band                    = self.args.band
        self.repeatCount             = self.args.RepeatCount
        self.polarization            = self.args.polarization
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.atmSubscanDuration      = self.args.atmSubscanDuration
        self.onOffSubscanDuration    = self.args.onOffSubscanDuration
        self.refAzOffset             = self.args.refAzOffset
        self.elLimit                 = self.args.ElLimit
        self.ambientLoad             = self.args.ambientLoad
        bbNameStr                    = self.args.bbNames
        lo1Str                       = self.args.LO1
        lo2Str                       = self.args.LO2
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)
        if lo1Str is not None and lo1Str != "":
            for s in lo1Str.split(','):
                try:
                    self.lo1List.append(1.0e9*float(s))
                except ValueError as ex:
                    msg = "Invalid number in LO1 list: '%s'" % s
                    ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    raise ex
        if lo2Str is not None and lo2Str != "":
            for s in lo2Str.split(','):
                try:
                    self.lo2List.append(1.0e9*float(s))
                except ValueError as ex:
                    msg = "Invalid number in LO2 list: '%s'" % s
                    ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    raise ex

    def setSource(self):
        self.source = self.sourceHelper.getBlankSkySource()

    def generateLO1List(self):
        if len(self.lo1List) > 0:
            self.logInfo("LO1 list provided by user: %s" % (str(self.lo1List)))
            return
        LOrange = self.bandLOrange[self.band]
        LOmin = LOrange[0]
        LOmax = LOrange[1]
        LOspan = LOmax - LOmin
        step = 3.0
        steps = int(round(LOspan / step))
        step = LOspan / steps
        ret = []
        for i in range(steps+1):
            ret.append(1.0e9 * (LOmin + i*step))
        self.logInfo("LO range: %.3f, %.3f; step = %.3f GHz" % (LOmin, LOmax, step))
        self.logInfo("LO1 list: %s" % (str(ret)))
        self.lo1List = ret

    def generateTunings(self):
        self._calSpectralSpecs = []
        self.generateLO1List()
        corrType = self._array.getCorrelatorType()
        for fLO1 in self.lo1List:
            #for band in self.bandList:
            fFLOOG = 32.5e6
            fLO2 = [9.979e9, 8.021e9, 8.021e9, 9.979e9]
            useUSB = [False, False, True, True]
            if self.band in [1, 9, 10]:
                fLO2 = [8.021e9, 10.021e9, 11.979e9, 13.979e9]
                useUSB = [True, True, True, True]
            if self.band == 6:
                fLO2 = [11.979e9, 10.021e9, 10.021e9, 11.979e9]
            # allow for user LO2 override
            if len(self.lo2List) > 0:
                for i in range(len(self.lo2List)):
                    if i >= len(fLO2):
                        break
                    fLO2[i] = self.lo2List[i]
            #
            # A bit ugly: make a normal SS, then turn it into a hardware one
            #
            frequency = fLO1
            if self.band == 1:
                # avoid it looking like we try to tune outside the band
                frequency = fLO1 + 8e9
            ss = self._tuningHelper.GenerateSpectralSpec(
                    band = self.band,
                    intent = "interferometry_continuum",
                    corrMode = "FDM",
                    frequency = frequency,
                    corrType = corrType,
                    # 180deg is to be more of a science-like test -- more possible things to go wrong!
                    enable180DegWalsh = False,
                    dualMode = True,
                    pol = self.polarization,
                    dump = self.dumpDuration,
                    channelAverage = self.channelAverageDuration,
                    integration = self.integrationDuration,
                    tpSampleTime = self.tpIntegrationDuration)
            ss.FrequencySetup.lO1Frequency.set(fLO1)
            ss.FrequencySetup.isUserSpecifiedLO1 = True
            ss.FrequencySetup.hasHardwareSetup = True
            ss.FrequencySetup.floog.set(fFLOOG)
            ss.FrequencySetup.tuneHigh = True
            ss.FrequencySetup.dopplerReference = u'topo'
            ss.name = "Band %d LO1=%.3f GHz" % (self.band, fLO1 / 1.0e9)
            for i in range(len(ss.FrequencySetup.BaseBandSpecification)):
                bbs = ss.FrequencySetup.BaseBandSpecification[i]
                bbs.lO2Frequency.set(fLO2[i])
                #bbs.centerFrequency.set(fSky[i])
                bbs.useUSB = useUSB[i]
                bbs.use12GHzFilter = False
                sb = "LSB"
                if useUSB[i]:
                    sb = "USB"
            Observation.SSRTuning.fillSkyFreqsFromLOs(ss, self)
            # set SpW sideband equal to signal path sideband
            Observation.SSRTuning.setSpWSBsFromSignalPathSBs(ss, self)
            ss.name = "Band %d hardware setup LO1=%.3fGHz" % (self.band, fLO1)
            self.logInfo("SpeectralSpec: %s " % ss.toxml())
            self._calSpectralSpecs.append(ss)

    def doAtmCal(self, ss):
        src = self.source
        self.logInfo("Now doing AtmCal scan")
        try:
            ss = Observation.SSRTuning.generateAtmSpectralSpec(ss)
            atm = Observation.AtmCalTarget.AtmCalTarget(src, ss, doHotLoad=True)
            atm.setOnlineProcessing(True)
            atm.setDataOrigin('FULL_RESOLUTION_AUTO')
            atm.setDoZero(False)
            atm.setSubscanDuration(self.atmSubscanDuration)
            atm.setIntegrationTime(1.5)
            atm.setWVRCalReduction(True)
            atm.setApplyWVR(False)
            self.logInfo('Executing AtmCal on ' + str(src.sourceName) + '...')
            atm.execute(self._obsmode)
            self.logInfo('Completed AtmCal on ' + str(src.sourceName))
        except BaseException as ex:
            msg = "Error executing AtmCal on source '%s'" % str(src.sourceName)
            self.logException(msg, ex)
            self.closeExecution(ex)
            raise ex

    def doOnOff(self, ss):
        src = self.source
        offSourcePointing = CCL.SourceOffset.stroke(math.radians(self.refAzOffset/3600.),0,0,0,Control.HORIZON)
        self.logInfo("Now doing the ON - OFF subscan sequence" )
        try:
            onoff = Observation.OnOffTarget.OnOffTarget(SubscanFieldSource = src,
                           SpectralSpec = ss,
                           SubscanDuration = self.onOffSubscanDuration,
                           IntegrationTime = 2.0 * self.onOffSubscanDuration * self.repeatCount,
                           sourceOffset = offSourcePointing)
            if self.ambientLoad:
                # CSV-3191: put ambient load in front of receiver
                self.logInfo('Beware: You are going to be observing with the Ambiant LOAD in front of your receivers. You should not see any coherent fringes.')
                onoff.setObserveWidget('AMBIENT_LOAD')
                onoff.setLevelWidget('AMBIENT_LOAD')
            self.logInfo('Executing OnOff on ' + str(src.sourceName) + '...')
            onoff.execute(self._obsmode)
            self.logInfo('Completed OnOff on ' + str(src.sourceName))
            if self.ambientLoad:
                # take it out as the execute does not do this 
                self._obsmode.setCalibrationDevice(PyDataModelEnumeration.PyCalibrationDevice.NONE)
                onoff.setObserveWidget('NONE')
        except BaseException as ex:
            msg = "Error executing cal survey scans on source %s" % src.sourceName
            self.logException(msg, ex)
            self.closeExecution(ex)
            raise ex

    def doScans(self):
        for ss in self._calSpectralSpecs:
            self.logInfo("Running scans for tuning: %s" % (str(ss.name)))
            self.doAtmCal(ss)
            self.doOnOff(ss)



obs = ObsCalSpectralCheckout()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    #obs.generateTunings()
    obs.setSource()
    obs.generateTunings()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing scans...")
obs.doScans()
obs.closeExecution()

