#! /usr/bin/env python
# ALMA - Atacama Large Millimeter Array
# (c) Associated Universities Inc., 2009 - 2014
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that i1t will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

#************************************************************************
#   NAME StandardInterferometry.py
# 
#   SYNOPSIS This script is intended to meet the needs of single or multi-target
#       observations which have a common spectral configuration for all
#       science targets and a shared primary phase calibrator.  
# 
#------------------------------------------------------------------------

#
# forcing global imports is due to an OSS problem
#
global CCL
from CCL.Global import getArray
import CCL.APDMSchedBlock
from CCL.ObservingModeBase import SBOutOfTime
from PyDataModelEnumeration import PyCalibrationDevice
global Observation
import Observation.SchedulingBlock
from Observation.SBExecutionMode import SBExecutionMode, SBExecutionError
from Observation.Global import simulatedArray, addClassToMode
from Observation.PointingCalTarget import PointingCalTarget
from Observation.ScanList import ScanList
from Observation.SBExecState import SBExecState
import Observation.SSRLogger
global TelCalParameters
import TelCalParameters

# Ugly hack to get nice logging when we're not written as a class
class StandardInterferometry(Observation.SSRLogger.SSRLogger):
    def __init__(self):
        Observation.SSRLogger.SSRLogger.__init__(self, self.__class__.__name__)
logger = StandardInterferometry()

sb = Observation.SchedulingBlock.SchedulingBlock()

logger.logInfo('Running %s scheduling block' % sb._controlSB.name)
# TODO: add type-check for expert parameters : PRTSPR-24831

# Get expert parameters as needed here
elLimit         = float(sb.getExpertParameter('ElLimit', default=20.))
maxPointingSeparation = float(sb.getExpertParameter('MaxPointingSeparation', default=25.))
sourceCycle     = float(sb.getExpertParameter('SourceCycleTime', default=0))
useScanSequence = bool(int(sb.getExpertParameter('useScanSequence', default=True)))
useDualMode     = bool(int(sb.getExpertParameter('useDualMode', default=True)))
fastSWEnabled   = bool(int(sb.getExpertParameter('FastSWEnabled', default=False)))
# Below are parameters for DiffGainCalTarget
dgcExtCycle   = float(sb.getExpertParameter('dgcExtCycle', default=7200.))
dgcIntCycle   = float(sb.getExpertParameter('dgcIntCycle', default=60.))
dgcRefIntTime = float(sb.getExpertParameter('dgcRefIntTime', default=240.))
dgcSciIntTime = float(sb.getExpertParameter('dgcSciIntTime', default=180.))
dgcSourceName = str(sb.getExpertParameter('dgcSourceName', default=''))
dgcSourceRa   = str(sb.getExpertParameter('dgcSourceRa', default=''))  # hh:mm:ss.sss
dgcSourceDec  = str(sb.getExpertParameter('dgcSourceDec', default='')) # dd.mm.ss.sss
dgcPostpone   = float(sb.getExpertParameter('dgcPostpone', default=0.))
useDGC        = int(sb.getExpertParameter('useDGC', default=0))
disableSBValidation  = int(sb.getExpertParameter('disableSBValidation', default=0))
useHarmonicB2B= bool(int(sb.getExpertParameter('useHarmonicB2B', default=1)))
harmonicB2BPreferredBand= int(sb.getExpertParameter('harmonicB2BPreferredBand', default=0))
atmSubscanDuration = int(sb.getExpertParameter('atmSubscanDuration', default=-1))
if harmonicB2BPreferredBand <= 0:
    harmonicB2BPreferredBand = None
# A parameter for overrding WVR correction on/off/both. ("", "off", "on", "both")
wvrProduct = str(sb.getExpertParameter("wvrProduct", default=''))
onSourceATM = bool(sb.getExpertParameter("onSourceATM", default=0))
# logger.logInfo('%s' % sb.getExpertParameters())

logger.logInfo("useDualMode value %s" % useDualMode)
logger.logInfo("useScanSequence value %s" % useScanSequence)
logger.logInfo('fastSWEnabled=%s' % fastSWEnabled)
logger.logInfo('wvrProduct=%s' % wvrProduct)
logger.logInfo("useHarmonicB2B = %s, harmonicB2BPreferredBand = %s" % (str(useHarmonicB2B), str(harmonicB2BPreferredBand)))

B_max = 450.0 # length, in meters, of the longest baseline in the array to be executed.  Could either
              # get this from the resolution requirement or from the scheduler with the configuration
if sourceCycle <= 0:
    sourceCycle = 12.0/(B_max*7.29e-5)
logger.logInfo('Source Cycle Time: %f s' % sourceCycle)
sessionControl = int(sb.getExpertParameter('SessionControl', default=0))
logger.logInfo('Session control %d' % sessionControl)
logger.logInfo('Simulated Array %s' % simulatedArray())

# Get reference to the array and start the exec block
array = getArray()

array.beginExecution()

# None means OK. If an error occurs the status is replaced, in the catch block, by the exception.
status = None
obsmode = None
try:
    sbChecker = Observation.SchedulingBlock.SBConfigurationChecker(sb)
    errors = sbChecker.checkInterferometricSB()
    if len(errors) > 0 and not disableSBValidation:
        raise Exception("SB validation failed [%s]" % (",".join(errors)))

    obsmode = array.getInterferometryObservingMode()

    # Add the SB execution methods
    addClassToMode(obsmode, SBExecutionMode)
    logger.logInfo("ASDM UID = %s" % (obsmode.getExecBlockUID()))

    # Dynamic triggeing of fast SW (experimental)
    dynamicFastSW = bool(sb.getExpertParameter("dynamicFastSW", default=0))
    logger.logInfo("dynamicFastSW = %s" % (dynamicFastSW))
    try:
        dynamicFC = Observation.SchedulingBlock.DynamicFastCycle(sb)
        dfc = dynamicFC.checkExecutable(obsmode)
        logger.logInfo("Executability with the current phase RMS: %s" % (dfc))
    except:
        import traceback
        self.logWarning(traceback.format_exc())
    if dynamicFastSW:
        if dfc == dynamicFC.FAST_SW_REQUIRED:
            msg = "Not executable with the default cycle time, but " + \
                  "executable with the fast cycling. Will update " + \
                  "cycle time"
            logger.logInfo(msg)

            newSubscand = 18.144
            newCycleTime = 54.
            dynamicFC.overwritePhaseReferencingCycleTime(newSubscand,
                                                         newCycleTime)

    # # If the script is executed from manual array, the script has to record
    # # SLT entry by itself
    # obsmode.recordShiftLogIfRequired(logger)

    # Enable Online WVR correction
    corrType = array.getCorrelatorType()
    enableWVRCorrection = False
    if str(corrType) == "BL":
        enableWVRCorrection = True
    arrayName = array._arrayName.replace("CONTROL/", "")
    logger.logInfo("Setting TelCal parameters for array %s: ontheflyWVRcorrection=%s spectrum=True useMiddleElevation=True" % (arrayName, enableWVRCorrection))
    if not simulatedArray():
        tcParameters = TelCalParameters.TelCalParameters(arrayName)
    else:
        from Observation.SimulatedTelCalParameters import SimulatedTelCalParameters
        tcParameters = SimulatedTelCalParameters(arrayName)

    tcParameters.setCalibParameter('ontheflyWVRcorrection', enableWVRCorrection)
    # spectrum is to enable channel-by-channel bandpass result. Also binningFactor can be specified if desired.
    tcParameters.setCalibParameter('spectrum', True)
    tcParameters.setCalibParameter('useMiddleElevation', True)

    # ICT-19478/ICT-19476
    from Observation.SSRConfigParameters import setTelCalRefAntennaList
    setTelCalRefAntennaList(tcParameters=tcParameters)

    # ICT-15218
    spec = sb.getRepresentativeTarget().getSpectralSpec()
    isAllTDM_ = Observation.SSRTuning.isAllTDM(spec)
    logger.logInfo("All TDM=%s" % (isAllTDM_))
    if isAllTDM_:
        tcParameters.setCalibParameter("binningFactor", 1)
    else:
        tcParameters.setCalibParameter("binningFactor", 2)

    # Session memory
    sbe = SBExecState(array, sb)
    # Clear out the state of the science stop button
    sbe.setStopButtonPressed(stop=False)
    isFirstExecutionInSession = False
    if sessionControl:
        sbe.startSession()
        isFirstExecutionInSession = sbe.isFirstTime()
        logger.logInfo("isFirstExecutionInSession=%s" % (isFirstExecutionInSession))
    obsmode.addSBExecState(sbe)

    # Just start on the sky, to be sure. Should be in the mode defaults.
    obsmode.setCalibrationDevice(PyCalibrationDevice.NONE)

    # Set observing mode restrictions
    logger.logInfo("Setting elevation limit to %f degrees" % elLimit)
    obsmode.setElevationLimit('%f deg' % elLimit)

    endTime = sb.getMaximumExecutionTime()
    logger.logInfo("Setting maximum execution time to %s seconds" % (endTime))
    obsmode.setMaximumSchedBlockTime(endTime)

    logger.logInfo("Setting max. pointing separation  to %f degrees" % maxPointingSeparation)
    PointingCalTarget.setMaximumSeparation('%f deg' % maxPointingSeparation )

    groupList = sb.getGroupList()
    # # Check the visibility of science targets in all groups.
    # for iGroup, group in enumerate(groupList):
    #     if obsmode.checkScienceTargetVisibility(group):
    #         continue
    #     raise SBExecutionError("No visible science target [Group%d]" % (iGroup + 1))

    for iGroup, group in enumerate(groupList):
        # Perform doppler correction for spectral specs associated with this group.
        sb.adjustDopplerByGroup(obsmode, group)

    # ICT-10545 Cycle-5 fudge to use OT auto-generated 90 deg switching SpectralSpecs
    Observation.SSRTuning.fiddle90degSpectralSpecs(sb, logger, obsmode)

    # ICT-9118
    if groupList[1].isBandToBand() and useHarmonicB2B:
        # At this moment, we use DiffGainCalTarget solely for B2B observation.
        # In SCIREQ-1657, it was requested to let SSR to decidede whether to
        # use harmonic band switching when the B2B mode is triggered.
        isUpdated = Observation.SSRTuning.overrideSpectralSpecsWithHarmonicB2B(sb, array, logger, obsmode, harmonicB2BPreferredBand)
        logger.logInfo("Overwritten spectral setups to use harmonic B2B: %s" %
                       (isUpdated))

        if isUpdated:
            # ICT-14764: Add another bandpass target to do bandpass calibration
            # in LF, if required
            groupList[0].appendReferenceBandpassTarget()
        else:
            msg = "There are no possible bands to use as B2B reference"
            logger.logInfo(msg)

    # In bands 8 and lower it makes most practical sense to point in-band if we
    # have weather for science in that band. The practical difficulties of
    # maintaining good TMCDB band offsets trump the S/N arguments, which are not
    # that great anyway as people forget the positive effect of decreasing beam
    # width with increasing frequency.
    Observation.SSRTuning.overridePointingSpectralSpecs(sb, array, logger, obsmode)

    if sb.isSpectralScanSB():
        # For Spectral Scan type SBs, we need to do Query on one Science Observing
        # Group and transfer the Query results to other Groups
        from Observation.SchedulingBlock import SpectralScanQueryResultMapper
        ssqw = SpectralScanQueryResultMapper(sb, obsmode, elLimit)
        ssqw.resolveQueries(isFirstExecutionInSession=isFirstExecutionInSession)

    for iGroup, group in enumerate(groupList):
        # ICT-7809: Estimate source flux and register to FieldSource.
        group.setSourceFluxForNonQuerySources()
        # Then, resolve queries.
        group.prepareTargets(minEl=elLimit,
                             isFirstExecutionInSession=isFirstExecutionInSession,
                             obsmode=obsmode)
        sb.updateTargetList()
        # And, create associated ATM targets (spectral specs shall be in TOPO)
        group.createAtmCalTargets(isSingleDish=False,
                                  atmSubscanDuration=atmSubscanDuration)

        # SBR targets are omitted from normal science SB executions, but
        # if it is requested to revert, here is the place to add SBR targets.
        ### group.createSBRatioCalTargets()

    sb.pruneSharableATMTargets()
    sb.updateTargetList()

    # Try to make sure there is at least two number of unique calibrator
    # sources observed in science setup.
    sb.checkForDuplicateCalSources()

    # If amp cal is missing, attach CALIBRATE_FLUX intent to bandpass.
    sb.tweakForMissingIntents()

    # Just for in case...
    for target in sb.getTargetList():
        dopRef = str(target.getSpectralSpec().FrequencySetup.dopplerReference)
        logger.logInfo("dopRef=%s getUseReferencePosition=%s %s" % (dopRef, target.getUseReferencePosition(), target))
        import Observation.AtmCalTarget
        if isinstance(target, Observation.AtmCalTarget.AtmCalTarget):
            assert(target.getUseReferencePosition() == True)
            if onSourceATM:
                target.setDoOnSource(True)
        else:
            target.setUseReferencePosition(False)

    if wvrProduct != "":
        # Switch between online-WVR on/off/both.
        sb.setWVRProduct(wvrProduct)

    # Assign an ID to each target.
    sb.setTargetsId()

    # If "AddToCleanup" target-level expert parameter is defined, add the
    # target to cleanup list.
    for group in groupList:
        group.addTargetsToCleanUpIfRequired(obsmode)

    if sessionControl:
        # Recycle Field Sources
        obsmode.sbe.recycleFieldSources()
        # Recycle attenuator settings
        # Mark _ifSkyAttenSet to re-use previous attenuator optimizations.
        # This relies on the fact that attenuator setting name is identical as
        # spectral spec name.
        obsmode.sbe.recycleAttenSettings()
        obsmode.sbe.restoreScienceTargetParameters()

    # construct DiffGainCalTarget, if needed
    logger.logInfo("useDGC=%s" % (useDGC))

    if useDGC:
        kwargs = dict(dgcExtCycle=dgcExtCycle,
                      dgcIntCycle=dgcIntCycle,
                      dgcRefIntTime=dgcRefIntTime,
                      dgcSciIntTime=dgcSciIntTime,
                      dgcSourceName=dgcSourceName,
                      dgcSourceRa=dgcSourceRa,
                      dgcSourceDec=dgcSourceDec,
                      dgcPostpone=dgcPostpone)
        isB2B = group.isBandToBand()
        logger.logInfo('isB2B = %s' % (isB2B))
        # Both B2B and BW switching observations can construct DGC target
        # instance in a same way.
        diffGainCalTarget = group.constructDiffGainCalTargetForB2B(obsmode,
                                                                   **kwargs)
        diffGainCalTarget.setOnlineProcessing(False)

        diffGainCalTargetList = [diffGainCalTarget]
    else:
        diffGainCalTargetList = sb.getDiffGainCalTargetList()

    useDGC = len(diffGainCalTargetList) > 0

    # Tweak reference position of targets, if they are specified in relative
    # horizontal frame.
    for group in sb.getGroupList():
        from Observation.ScienceTarget import ScienceTarget
        from Observation.AtmCalTarget import AtmCalTarget
        for target in group.getTargetList():
            if isinstance(target, ScienceTarget):
                continue
            for assocTarget in target.getAssociatedCalTarget():
                if not isinstance(assocTarget, AtmCalTarget):
                    continue
                assocTarget.tweakReferenceOffset(verbose=True,
                                                 saveOffsetToFS=True)

    # SCIREQ-290: remove ampcal, if it shares same source with bandpass cal
    for group in groupList:
        group.removeSharableTargets()

    # ICT-13002: Check the visibility of science targets in group2.
    # Naively assume that if group2 targets are ready, then targets in
    # the subsequent groups are also ready.
    estimatedGroup1Duration = sb.estimateGroup1Duration()
    for group in groupList[1:2]:
        if obsmode.checkScienceTargetVisibility(group,
                                                inTime=estimatedGroup1Duration):
            continue
        raise SBExecutionError("No visible science target [Group%d]" % (iGroup + 1))

    # The first group is for SB calibrators, we don't expect any science target.
    group = groupList[0]
    groupNumber = group.groupIndex
    logger.logInfo("Start Observing Group %d (Calibrations)" % groupNumber)
    scanList = ScanList() if useScanSequence else None

    # Optionally force-enable dual mode on SpectralSpecs.
    if useDualMode :
        for target in sb.getTargetList() :
            ss = target._spectralSpec
            logger.logInfo("Dual mode addition: evaluating SpectralSpec named '%s'" % str(ss.name))
            if not hasattr(ss,'SquareLawSetup') or ss.SquareLawSetup is None:
                logger.logInfo("Dual mode addition: adding SquareLawSetup to SpectralSpec named '%s'" % str(ss.name))
                ss.SquareLawSetup = CCL.APDMSchedBlock.SquareLawSetup()
                ss.SquareLawSetup.integrationDuration.set(0.016)
    # ICT-5552: For Cycle-3 OT will still give us short subscans, but we'll combine them when we can
    for target in sb.getTargetList() :
        logger.logInfo("setting target.coalesceSubscans = True on '%s' it was %s" % (str(target), str(target.coalesceSubscans)))
        target.coalesceSubscans = True
    # From Cycle-6 disable online WVR if it was enabled, e.g. for Cycle-5 carry-over
    for target in sb.getTargetList() :
        ss = target._spectralSpec
        logger.logInfo("APC check: evaluating SpectralSpec named '%s'" % str(ss.name))
        if ss.hasBLCorrelatorConfiguration():
            logger.logInfo("APC check: setting aPCDataSets = AP_UNCORRECTED for BL configuration in '%s'" % str(ss.name))
            ss.BLCorrelatorConfiguration.aPCDataSets = u'AP_UNCORRECTED'

    obsmode.executeCalTargetList(group.getFocusCalTargetList())

    # Polarization section needs to be cleanup a bit.
    obsmode.executeCalTargetList(group.getPolarizationCalTargetList(),
                                 deferToCleanupList=True, scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)
    if len(group.getPolarizationCalTargetList()) > 0:
        obsmode.addTargetToCleanupList(group.getPolarizationCalTargetList()[0])

    # enable online bandpass
    for target in sb.getBandpassCalTargetList():
        logger.logInfo("Online bandpass: enable online processing on target '%s'" % str(target))
        target.setOnlineProcessing(True)

    import Observation.CalParameterOptimization
    cParamOpt = Observation.CalParameterOptimization.CalParameterOptimization()
    for target in group.getBandpassCalTargetList():
        cParamOpt.optimizeDuration(target, "bandpass", obsmode=obsmode)

    obsmode.executeCalTargetListSpec(group.getBandpassCalTargetList(),
                                     deferToCleanupList=True,
                                     scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)

    obsmode.executeCalTargetList(group.getDelayCalTargetList(),
                                 scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)

    obsmode.executeCalTargetList(group.getAmplitudeCalTargetList(),
                                 deferToCleanupList=True, scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)

    values = (obsmode.getElapsedTime(), estimatedGroup1Duration)
    msg = "Group1 End: Elapsed=%6.1f [sec] Estimated=%6.1f [sec]" % values
    logger.logInfo(msg)

    # Now the rest of the groups should contain science targets.
    for group in groupList[1:]:
        # TODO: For Cycle5?: allow self-calibration

        for target in group.getPhaseCalTargetList():
            cParamOpt.optimizeDuration(target, "phase", obsmode=obsmode)

        groupNumber = group.groupIndex
        # The primary phase calibrator is assumed to be the one with the
        # shortest cycle time in the group.  This effectivly sets the scan
        # duration.
        primaryPhaseCal = obsmode.getPrimaryPhaseCal(group)
        if primaryPhaseCal is None:
            logger.logError("No phase calibrator available in Observing Group %s" % groupNumber)
            raise SBExecutionError("There is no available phase calibrator, at this moment.")
        else:
            logger.logInfo("Selected %s as Primary phase calibrator" % primaryPhaseCal)

        # ICT-7246: If phase cycle time is less than 150 seconds, enable fast SW mode.
        fastSWEnabled = group.isFastSWRequired()
        logger.logInfo("[Group%d] fastSWEnabled=%s" % (groupNumber, fastSWEnabled))
        logger.logInfo("Start Observing Group %d (Science)" % groupNumber)

        # A group is defined to be complete when all science targets within
        # the group are complete (or unobservable)
        obsmode.assignSubcycleTime(group.getScienceTargetList(), sourceCycle)
        while not group.isComplete(obsmode):
            # Just for reporting the achieved integration times
            sb.checkRemainingIntegrationTimes()

            if obsmode.sbe.getStopButtnPressed():
                # ICT-15045
                status = KeyboardInterrupt("science stop button pressed")
                logger.logInfo("'science' stop button pressed")
                break

            outerCycleTargets = list(group.getFocusCalTargetList()) + \
                                list(group.getAmplitudeCalTargetList()) + \
                                list(group.getPolarizationCalTargetList()) + \
                                list(group.getBandpassCalTargetList()) + \
                                list(group.getDelayCalTargetList()) + \
                                list(group.getCheckSourceCalTargetList())
            if useDGC:
                outerCycleTargets.extend(diffGainCalTargetList)

            secondaryPhaseCals = list(group.getPhaseCalTargetList())
            secondaryPhaseCals.remove(primaryPhaseCal)

            # If one of these target are executed, it would be nice
            # to call primary.overwriteNextExecutionTime(value=0.),
            # since in that case, we can make that duration of
            # each science target execution more uniform.
            obsmode.executeCalTargetList(group.getFocusCalTargetList(),
                                         scanList=scanList)
            if useScanSequence:
                scanList.execute(obsmode)

            obsmode.executeCalTargetList(group.getPolarizationCalTargetList(),
                                         scanList=scanList)
            if useScanSequence:
                scanList.execute(obsmode)

            obsmode.executeCalTargetList(group.getBandpassCalTargetList(),
                                         scanList=scanList)
            if useScanSequence:
                scanList.execute(obsmode)

            obsmode.executeCalTargetList(group.getAmplitudeCalTargetList(),
                                         scanList=scanList)
            if useScanSequence:
                scanList.execute(obsmode)

            if useDGC:
                # DiffGainCal
                # For B2B observations, it is desired to execute DGC just before the primary phase cal execution.
                for diffGainCalTarget in diffGainCalTargetList:
                    isExecuted = diffGainCalTarget.executeIfNeeded(obsmode, scanList=scanList)
                    if isExecuted:
                        # Force execution of next primary phase calibrator.
                        primaryPhaseCal.overwriteNextExecutionTime(value=0.)

                if useScanSequence:
                    scanList.execute(obsmode)

            obsmode.executeCalTargetList(group.getPhaseCalTargetList(),
                                         deferToCleanupList=True,
                                         minimumInCleanupList=len(group.getPhaseCalTargetList()),
                                         scanList=scanList)
            if not fastSWEnabled and useScanSequence:
                scanList.execute(obsmode)

            # Execute the ATM cal target for the science target. Note that
            # there is only one science target that has the associated ATM
            # target.
            atmCalTargets = []
            for target in group.getScienceTargetList():
                atmCalTargets.extend(target.getAssociatedAtmCalTarget())
            timeToNextAtm = \
                primaryPhaseCal.getTimeUntilNextCalibrator(obsmode,
                                                           atmCalTargets,
                                                           scanList=scanList)
            executed = False
            for atmTarget in atmCalTargets:
                executed |= atmTarget.executeIfNeeded(obsmode)

            if executed and primaryPhaseCal.getCycleTime() <= 60:
                # A bit tricky for ICT-19519

                # When fastSWEnabled, we want to bracket the ATM scan
                # with phase scans. For the first iteration in this
                # while-loop, phase scan is sequence in the scanList.
                # For the subsequent iterations, ScanList should be
                # emplty. If it is emplty, primary phase calibrator's
                # executions is forced.
                if len(scanList.getScanSequenceSpecification()) == 0:
                    primaryPhaseCal.execute(obsmode, scanList=scanList)
                    if not fastSWEnabled and useScanSequence:
                        scanList.execute(obsmode)

            # Check-source execution
            checkSources = group.getDelayCalTargetList() +\
                           group.getCheckSourceCalTargetList()
            timeToNextChk = \
                primaryPhaseCal.getTimeUntilNextCalibrator(obsmode,
                                                           checkSources,
                                                           scanList=scanList)
            if timeToNextChk is not None and timeToNextChk < 0:
                # Check-Source execution.
                # It is executed at every outer loop execution on the target.
                obsmode.executeCalTargetList(checkSources, scanList=scanList)
                if not fastSWEnabled and useScanSequence:
                    scanList.execute(obsmode)

                # ICT-19519
                if primaryPhaseCal.getCycleTime() <= 60:
                    primaryPhaseCal.execute(obsmode, scanList=scanList)
                    if not fastSWEnabled and useScanSequence:
                        scanList.execute(obsmode)

            while not group.isComplete(obsmode):
                # This is going to be compared with ATM's next requested time,
                # which is not sessionized.
                currentElapsedTime = obsmode.getElapsedTime()
                if useScanSequence:
                    currentElapsedTime += scanList.getCurrentCompletionTime()
                nextReqTimes = [target.getNextRequiredTime() for target in atmCalTargets]
                if len(nextReqTimes) > 0 and min(nextReqTimes) <= (currentElapsedTime - 0):
                    msg = "Stop queueing because ATM calibration is needed"
                    logger.logInfo(msg)
                    break

                # Inner-cycle: if scan-sequence is enabled, try to sequence as many as possible science and phase scans.
                # execute science targets until either a calibration is needed
                obsmode.executeScienceTargetList(group,
                                                 outerCycleTargets + [primaryPhaseCal],
                                                 scanList=scanList)
                if not fastSWEnabled and useScanSequence:
                    scanList.execute(obsmode)
                # then, phase-cal
                primaryPhaseCal.execute(obsmode, scanList=scanList)

                if not fastSWEnabled and useScanSequence:
                    scanList.execute(obsmode)

                # Check whether the primary phase calibrator will be above the
                # elevation limit for the next cycle.
                # ICT-6300: If cycle time is extremely long, cap the duration
                # for visibility check by the remaining science target
                # integration time.
                duration = min(primaryPhaseCal.getCycleTime(),
                               group.getRemainingIntegrationTime() + \
                               primaryPhaseCal.getIntegrationTime())
                if not primaryPhaseCal.isObservable(obsmode, duration=duration,
                                                    scanList=scanList):
                    primaryPhaseCal = obsmode.getPrimaryPhaseCal(group, primaryPhaseCal)
                    if primaryPhaseCal is None:
                        raise Exception('There is no available phase calibrator in group%d at this moment, and thus stopping this SB execution. If SB does contain at least one valid phase calibrator in group%d, then it is likely that all of them are (judged to be) below the elevation limit.' % (groupNumber, groupNumber))

                if not fastSWEnabled:
                    # If fast-sw disabled, do not try to sequence phase and science targets
                    break

                if useScanSequence and scanList.getCurrentCompletionTime() > 900:
                    break

                # loop-end condition. (If outer-cycle targets need execution, get out of inner loop)
                currentElapsedTime = obsmode.sbe.getElapsedTime() if sessionControl else obsmode.getElapsedTime()
                if useScanSequence:
                    currentElapsedTime += scanList.getCurrentCompletionTime()
                nextReqTimes = [target.getNextRequiredTime() for target in outerCycleTargets + secondaryPhaseCals]
                # primaryPhaseCal.getTimeUntilNextCalibrator(obsmode, outerCycleTargets + secondaryPhaseCals, scanList=scanList)
                if len(nextReqTimes) > 0 and min(nextReqTimes) <= currentElapsedTime:
                    msg = "stop queueing because check source is needed"
                    logger.logInfo(msg)
                    break

            # Exexcute sequence of science and primary phase cal scans
            if useScanSequence:
                scanList.execute(obsmode)

            if obsmode.sbe.getStopButtnPressed():
                # ICT-15045
                status = KeyboardInterrupt("science stop button pressed")
                logger.logInfo("'science' stop button pressed")
                break

        # Group is complete: remove the primaryPhaseCal from cleanup:
        logger.logInfo("Observing Group %s is completed." % groupNumber)
        obsmode.removeTargetFromCleanupList(primaryPhaseCal)

        # ICT-8368: 'cleanup' execution for secondary phase calibrators
        # should be made at end of the group, not at the end of execution
        for target in secondaryPhaseCals:
            target.execute(obsmode)
            obsmode.removeTargetFromCleanupList(target)

    # All groups are complete: execute the cleanup (e.g. bandpass or amplitude cals)
    logger.logInfo("All groups are completed.")
    logger.logInfo("Executing cleanup list.")
    obsmode.executeCleanupList(scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)
except SBOutOfTime:
    logger.logWarning("Running short of time, forcing cleanup!")
    groupNumber = group.groupIndex
    if sessionControl and groupNumber > 0 and hasattr(group, "_subcycleIndex"):
        currentIndex = sbe.getScienceTargetIndex(groupNumber)
        if currentIndex == group._subcycleIndex:
            group._subcycleIndex = (group._subcycleIndex + 1) % len(group.getScienceTargetList())
        sbe.saveScienceTargetIndex(groupNumber, group._subcycleIndex)
    if useScanSequence:
        scanList.execute(obsmode)
    logger.logInfo("Executing cleanup list.")
    obsmode._maximumTime = None
    obsmode.executeCleanupList(scanList=scanList)
    if useScanSequence:
        scanList.execute(obsmode)

except Exception as ex:
    logger.logError("Exception caught: %s" % ex)
    status = ex
    raise

finally:
    try:
        tcParameters.setCalibParameter("binningFactor", 1)
    except:
        import traceback
        logger.logError(traceback.format_exc())
        logger.logError("Could not reset the binning factor")

    if sessionControl:
        sbe.endSBExecution()
    try:
        obsmode.resetLimits()
    except:
        logger.logError("Could not reset limits")

    array.endExecution(status)
    try:
        if obsmode:
            obsmode.cleanupObsMode(array, logger)
    except:
        import traceback
        logger.logWarning("Failed to execute cleanupObsMode()")
        logger.logWarning(traceback.format_exc())
#
# ___oOo___
