#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalIFDelays.py 214237 2015-02-06 05:16:28Z nphillip $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global random
import random

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.PowerLevelCalTarget
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalPowerLevels(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("RepeatCount", int, 2),
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 0.576),
        #Observation.ObsCalBase.scriptOption("AtmIntegrationDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("NumTargets", int, 10),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 3),
        Observation.ObsCalBase.scriptOption("bbNames", str, "")
    ]

    def parseOptions(self):
        self.repeatCount             = self.args.RepeatCount
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.atmSubscanDuration      = self.args.AtmSubscanDuration
        self.sbrSubscanDuration      = self.args.SBRSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        #self.atmIntegrationDuration  = self.args.AtmIntegrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.numTargets              = self.args.NumTargets
        self.pointFocusBand          = self.args.pointFocusBand
        bbNameStr                    = self.args.bbNames
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        freq = Observation.SSRTuning.bandFreqs_delayMeasurement[self.pointFocusBand]
        self.ss = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                frequency = freq,
                sameBBFreqs = True,
                corrType = corrType,
                bbNames = self.bbNames,
                dualMode = True,
                #autoOnly = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration)

    def setAttenuatorSettings(self):
        calResults = self._array.getCalResults()
        self.attSettingNames = []
        for ifsAtt in range(16):
            for bbAtt2 in range(64):
               bbAtt = 0.5*bbAtt2
               name = "IFS %2d dB IFProc %4.1f dB" % (ifsAtt, bbAtt)
               self.attSettingNames.append(name)
               self.logInfo("Creating and storing in CalResults attenuator setting '%s'" % name)
               bbAtts = [bbAtt, bbAtt, bbAtt, bbAtt]
               att = Control.AttenuatorSetting(ifProcPol0=bbAtts, ifProcPol1=bbAtts,
                       ifswPol0USB=ifsAtt, ifswPol1USB=ifsAtt, ifswPol0LSB=ifsAtt, ifswPol1LSB=ifsAtt)
               antAtt = []
               for ant in self._array.antennas():
                   antAtt.append(Control.NamedAttenuatorSetting(ant, att))
               calResults.setAttenuatorSettings(name, antAtt)
        random.shuffle(self.attSettingNames)


    
    def doPowerLevelCals(self):
        src = self._srcPointFocus
        try:
            powerCal = Observation.PowerLevelCalTarget.PowerLevelCalTarget(src, self.ss, attSettingNames=self.attSettingNames)
            powerCal.setSubscanDuration(4.0)
            powerCal.setIntegrationTime(1.0)
            self.logInfo('Executing PowerLevelCal on ' + src.sourceName + '...')
            powerCal.execute(self._obsmode)
            self.logInfo('Completed PowerLevelCal on ' + src.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing cal survey scans on source %s" % src.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex




obs = ObsCalPowerLevels()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.setAttenuatorSettings()
    obs.findPointFocusSource()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing PowerLevelCals...")
obs.doPowerLevelCals()
obs.closeExecution()

