
# Application directory for writing configuration and log files
DEFAULT_ALMA_OT_PATH="$HOME/.almaot"
# Default system wide java command before we try searching for a better one
DEFAULT_JRE_PATH="java"

# Foreground colours
BLACK="\033[0;30m"
RED="\033[0;31m"
GREEN="\033[0;32m"
YELLOW="\033[0;33m"
BLUE="\033[0;34m"
MAGENTA="\033[0;35m"
CYAN="\033[0;36m"
WHITE="\033[1;37m"
RESET="\033[0m"

# Functions
# ---------

print_usage () {
    echo -e "Usage: ${BLUE}$(basename ${0})${RESET} [${GREEN}OPTIONS${RESET}]"
    echo -e ""
    echo -e "Options:"
    echo -e ""
    echo -e "  ${GREEN}-h, --help${RESET}                Print this help description"
    echo -e "  ${GREEN}-j, --jre-path PATH${RESET}       Java Runetime Environment Path (default ${DEFAULT_JRE_PATH})"
    echo -e "  ${GREEN}-m, --max-heap-size SIZE${RESET}  Maximum heap space size in G (default ${DEFAULT_MAX_HEAP_SIZE})"
}

# Main script body
# ----------------

jre_path="$DEFAULT_JRE_PATH"
max_heap_size="$DEFAULT_MAX_HEAP_SIZE"

# Parse command line options
while [[ $# -gt 0 ]]; do
    case $1 in
        -h|--help)
            print_usage
            exit 0
            ;;
        -j|--jre-path)
            jre_path="$2"
            shift # past argument
            shift # past value
            ;;
        -m|--max-heap-size)
            max_heap_size="$2"
            shift # past argument
            shift # past value
            ;;
        *)
            positional_args+=("$1") # save positional arg
            shift # past argument
            ;;
    esac
done
# Restore positional parameters
set -- "${positional_args[@]}"

# Locate the JRE we are going to use for running the ALMA OT
java_cmd="$DEFAULT_JRE_PATH"
if [ "$jre_path" == "$DEFAULT_JRE_PATH" ]; then
    # An array of the various directories which could hold JREs for the various
    # platforms if a JRE is included with the release
    jre_path_array=("${INSTALL_PATH}/jdk-17.0.14+7-jre/bin/java")
    for jre_path in "${jre_path_array[@]}"
    do
        if [ -f "$jre_path" ]; then
            java_cmd="$jre_path"
            break
        fi
    done
fi

if ! command -v "$java_cmd" >> /dev/null 2>&1
then
    echo "Unable to find Java Runtime Environment (JRE) for $java_cmd"
    exit 1
fi

if [ ! -d "$DEFAULT_ALMA_OT_PATH" ]; then
    echo "Creating ALMA OT application directory $DEFAULT_ALMA_OT_PATH"
    mkdir -p "$DEFAULT_ALMA_OT_PATH"
fi

echo "Starting the ALMA OT application..."
echo "PLEASE NOTE:"
echo "If you are planning to work on large proposals (e.g. including many"
echo "sources and/or science goals) or on a Large Program you may consider"
echo "increasing the maximum heap space memory size. This will improve the"
echo "performance and usability of the ALMA OT application."
echo ""
echo "A default maximum of $max_heap_size G has been allocated for the ALMA OT application."
echo "You can increase the maximum heap size by using the --max-heap-size command line"
echo "option. In this example we increase the maximum heap space memory size to 16 G. You"
echo "can set it to higher values but please keep in mind the maximum memory available on"
echo "your computer:"
echo ""
echo "   bash ./$(basename ${0}) --max-heap-size 16"
echo ""
echo "Using Java JRE from $(command -v $java_cmd)"
echo "ALMA OT log files are stored at $DEFAULT_ALMA_OT_PATH"

$java_cmd $DEFAULT_PROXY_SETTINGS \
    -cp "$INSTALL_PATH/lib/jACSUtil.jar:$INSTALL_PATH/lib/*" \
    --add-opens java.xml/com.sun.org.apache.xml.internal.serialize=ALL-UNNAMED \
    -Dacs.system.classpath.jardirs="$INSTALL_PATH/lib" \
    -Djava.util.logging.config.class=alma.obsprep.util.LoggingSetup \
    -Dalma.obsprep.PrintStackTrace=true \
    -DTelescope="Cycle12" \
    -Dot.submission.host="https://cycle-12.sps.alma.cl" \
    -Xmx${max_heap_size}g \
    alma.obsprep.ot.gui.toplevel.ObservingTool \
    2> "$DEFAULT_ALMA_OT_PATH/ot.log" 1>&2

exit $?
