/*
 * Decompiled with CFR 0.152.
 */
package skyview.geometry;

import java.util.Arrays;
import java.util.HashMap;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCardException;
import skyview.geometry.Converter;
import skyview.geometry.CoordinateSystem;
import skyview.geometry.Distorter;
import skyview.geometry.Projection;
import skyview.geometry.Rotater;
import skyview.geometry.Scaler;
import skyview.geometry.TransformationException;
import skyview.geometry.Transformer;
import skyview.geometry.distorter.DSS;
import skyview.geometry.distorter.Neat;

public class WCS
extends Converter {
    private double wcsScale;
    private CoordinateSystem csys;
    private Projection proj;
    private Distorter distort;
    private Scaler scale;
    int[] headerNaxis;
    private boolean stdWCS = true;
    private HashMap<String, Object> wcsKeys;
    private static boolean preferDSS = false;
    private int lonAxis = -1;
    private int latAxis = -1;
    private Header h;

    public WCS(CoordinateSystem csys, Projection proj, Scaler scale) throws TransformationException {
        this.csys = csys;
        this.proj = proj;
        this.scale = scale;
        this.add(csys.getSphereDistorter());
        this.add(csys.getRotater());
        this.add(proj.getRotater());
        this.add(proj.getProjecter());
        this.add(proj.getDistorter());
        this.add(scale);
        this.setWCSScale(scale);
    }

    public static void setPreferDSS(boolean flag) {
        preferDSS = flag;
    }

    public CoordinateSystem getCoordinateSystem() {
        return this.csys;
    }

    public Projection getProjection() {
        return this.proj;
    }

    public Scaler getScaler() {
        return this.scale;
    }

    public Distorter getDistorter() {
        return this.distort;
    }

    private void setWCSScale(Scaler s) {
        double[] p = s.getParams();
        double det = p[2] * p[5] - p[3] * p[4];
        this.wcsScale = 1.0 / Math.sqrt(Math.abs(det));
    }

    public WCS(Header h) throws TransformationException {
        this.wcsKeys = new HashMap();
        this.h = h;
        this.headerNaxis = new int[2];
        if (this.checkDSS()) {
            this.headerNaxis[0] = h.getIntValue("NAXIS1");
            this.headerNaxis[1] = h.getIntValue("NAXIS2");
            if (this.headerNaxis[0] == 0) {
                this.headerNaxis[0] = h.getIntValue("XPIXELS");
                this.headerNaxis[1] = h.getIntValue("YPIXELS");
            }
            this.doDSSWCS();
            this.stdWCS = false;
        } else if (this.checkNeat()) {
            this.headerNaxis[0] = h.getIntValue("NAXIS1");
            this.headerNaxis[1] = h.getIntValue("NAXIS2");
            this.doNeatWCS();
            this.stdWCS = false;
        } else {
            this.getAxes();
            if (this.lonAxis == -1 || this.latAxis == -1) {
                throw new TransformationException("Unable to find coordinate axes");
            }
            this.headerNaxis[0] = h.getIntValue("NAXIS" + this.lonAxis);
            this.headerNaxis[1] = h.getIntValue("NAXIS" + this.latAxis);
            this.extractCoordinateSystem();
            this.extractProjection();
            this.extractScaler();
        }
    }

    public boolean standardWCS() {
        return this.stdWCS;
    }

    public int[] getHeaderNaxis() {
        return this.headerNaxis;
    }

    public void setHeaderNaxis(int[] newAxes) {
        this.headerNaxis = new int[newAxes.length];
        System.arraycopy(newAxes, 0, this.headerNaxis, 0, newAxes.length);
    }

    public WCS(Header h, int lonAxis, int latAxis) throws TransformationException {
        this.lonAxis = lonAxis;
        this.latAxis = latAxis;
        this.extractCoordinateSystem();
        this.extractProjection();
        this.extractScaler();
    }

    private void getAxes() {
        int naxes = this.h.getIntValue("NAXIS");
        for (int i = 1; i <= naxes; ++i) {
            String axis = this.h.getStringValue("CTYPE" + i);
            if (axis == null || axis.length() < 4) continue;
            axis = axis.substring(0, 4);
            if (this.lonAxis == -1 && (axis.equals("RA--") || axis.equals("GLON") || axis.equals("ELON") || axis.equals("HLON"))) {
                this.lonAxis = i;
            }
            if (this.latAxis != -1 || !axis.equals("DEC-") && !axis.equals("GLAT") && !axis.equals("ELAT") && !axis.equals("HLAT")) continue;
            this.latAxis = i;
        }
        if (this.lonAxis > -1) {
            this.wcsKeys.put("CTYPE1", this.h.getStringValue("CTYPE" + this.lonAxis));
        }
        if (this.latAxis > -1) {
            this.wcsKeys.put("CTYPE2", this.h.getStringValue("CTYPE" + this.latAxis));
        }
    }

    private void extractScaler() throws TransformationException {
        double crpix1 = this.h.getDoubleValue("CRPIX" + this.lonAxis, Double.NaN);
        double crpix2 = this.h.getDoubleValue("CRPIX" + this.latAxis, Double.NaN);
        this.wcsKeys.put("CRPIX1", crpix1);
        this.wcsKeys.put("CRPIX2", crpix2);
        if (Double.isNaN(crpix1) || Double.isNaN(crpix2)) {
            throw new TransformationException("CRPIXn not found in header");
        }
        if (this.extractScaler1(crpix1 -= 0.5, crpix2 -= 0.5)) {
            return;
        }
        if (this.extractScaler2(crpix1, crpix2)) {
            return;
        }
        throw new TransformationException("No scaling information found in FITS header");
    }

    private boolean extractScaler1(double crpix1, double crpix2) throws TransformationException {
        double m22;
        double m21;
        double m12;
        double m11;
        boolean matrix = false;
        double cdelt1 = this.h.getDoubleValue("CDELT" + this.lonAxis, Double.NaN);
        double cdelt2 = this.h.getDoubleValue("CDELT" + this.latAxis, Double.NaN);
        this.wcsKeys.put("CDELT1", cdelt1);
        this.wcsKeys.put("CDELT2", cdelt2);
        if (Double.isNaN(cdelt1) || Double.isNaN(cdelt2)) {
            return false;
        }
        double crota = this.h.getDoubleValue("CROTA" + this.latAxis, Double.NaN);
        if (!Double.isNaN(crota) && crota != 0.0) {
            this.wcsKeys.put("CROTA2", crota);
            crota = Math.toRadians(crota);
            m11 = Math.cos(crota);
            m12 = Math.sin(crota);
            m21 = -Math.sin(crota);
            m22 = Math.cos(crota);
            matrix = true;
        } else {
            m11 = this.h.getDoubleValue("PC" + this.lonAxis + "_" + this.lonAxis, Double.NaN);
            boolean bl = matrix = !Double.isNaN(m11 + (m12 = this.h.getDoubleValue("PC" + this.lonAxis + "_" + this.latAxis, Double.NaN)) + (m21 = this.h.getDoubleValue("PC" + this.latAxis + "_" + this.lonAxis, Double.NaN)) + (m22 = this.h.getDoubleValue("PC" + this.latAxis + "_" + this.latAxis, Double.NaN)));
            if (matrix) {
                this.wcsKeys.put("PC1_1", m11);
                this.wcsKeys.put("PC1_2", m12);
                this.wcsKeys.put("PC2_1", m21);
                this.wcsKeys.put("PC2_2", m22);
            }
        }
        cdelt1 = Math.toRadians(cdelt1);
        cdelt2 = Math.toRadians(cdelt2);
        Scaler s = !matrix ? new Scaler(-cdelt1 * crpix1, -cdelt2 * crpix2, cdelt1, 0.0, 0.0, cdelt2) : new Scaler(-cdelt1 * (m11 * crpix1 + m12 * crpix2), -cdelt2 * (m21 * crpix1 + m22 * crpix2), cdelt1 * m11, cdelt1 * m12, cdelt2 * m21, cdelt2 * m22);
        s = s.inverse();
        if (this.lonAxis > this.latAxis) {
            s.interchangeAxes();
        }
        this.scale = s;
        this.add(s);
        this.setWCSScale(s);
        return true;
    }

    private boolean extractScaler2(double crpix1, double crpix2) throws TransformationException {
        double m22;
        double m21;
        double m12;
        boolean matrix;
        double m11 = this.h.getDoubleValue("CD" + this.lonAxis + "_" + this.lonAxis, Double.NaN);
        boolean bl = matrix = !Double.isNaN(m11 + (m12 = this.h.getDoubleValue("CD" + this.lonAxis + "_" + this.latAxis, Double.NaN)) + (m21 = this.h.getDoubleValue("CD" + this.latAxis + "_" + this.lonAxis, Double.NaN)) + (m22 = this.h.getDoubleValue("CD" + this.latAxis + "_" + this.latAxis, Double.NaN)));
        if (!matrix) {
            return false;
        }
        this.wcsKeys.put("CD1_1", m11);
        this.wcsKeys.put("CD1_2", m12);
        this.wcsKeys.put("CD2_1", m21);
        this.wcsKeys.put("CD2_2", m22);
        m11 = Math.toRadians(m11);
        m12 = Math.toRadians(m12);
        m21 = Math.toRadians(m21);
        m22 = Math.toRadians(m22);
        Scaler s = new Scaler(-m11 * crpix1 - m12 * crpix2, -m21 * crpix1 - m22 * crpix2, m11, m12, m21, m22);
        s = s.inverse();
        if (this.lonAxis > this.latAxis) {
            s.interchangeAxes();
        }
        this.scale = s;
        this.add(s);
        this.setWCSScale(s);
        return true;
    }

    private void extractCoordinateSystem() throws TransformationException {
        CoordinateSystem coords;
        String lonType = this.h.getStringValue("CTYPE" + this.lonAxis).substring(0, 4);
        String latType = this.h.getStringValue("CTYPE" + this.latAxis).substring(0, 4);
        String coordSym = null;
        if (lonType.equals("RA--") && latType.equals("DEC-")) {
            coordSym = this.frame() + this.equinox();
        } else {
            if (lonType.charAt(0) != latType.charAt(0)) {
                throw new TransformationException("Inconsistent axes definitions:" + lonType + "," + latType);
            }
            if (lonType.equals("GLON")) {
                coordSym = "G";
            } else if (lonType.equals("ELON")) {
                coordSym = "E" + this.equinox();
            } else if (lonType.equals("HLON")) {
                coordSym = "H" + this.equinox();
            }
        }
        this.csys = coords = CoordinateSystem.factory(coordSym);
        this.add(coords.getSphereDistorter());
        this.add(coords.getRotater());
    }

    private double equinox() {
        double equin = this.h.getDoubleValue("EQUINOX", Double.NaN);
        if (Double.isNaN(equin)) {
            equin = this.h.getDoubleValue("EPOCH", Double.NaN);
        }
        if (Double.isNaN(equin)) {
            equin = 2000.0;
        }
        this.wcsKeys.put("EQUINOX", 2000);
        return equin;
    }

    private String frame() {
        String sys = this.h.getStringValue("RADESYS");
        if (sys != null) {
            this.wcsKeys.put("RADESYS", sys);
        }
        if (sys == null) {
            double equin = this.equinox();
            if (equin >= 1984.0) {
                return "J";
            }
            return "B";
        }
        if (sys.length() > 3 && sys.substring(0, 3).equals("FK4")) {
            return "B";
        }
        return "J";
    }

    private void extractProjection() throws TransformationException {
        String latType;
        Projection proj = null;
        Scaler ncpScale = null;
        String lonType = this.h.getStringValue("CTYPE" + this.lonAxis).substring(5, 8);
        if (!lonType.equals(latType = this.h.getStringValue("CTYPE" + this.latAxis).substring(5, 8))) {
            throw new TransformationException("Inconsistent projection in FITS header: " + lonType + "," + latType);
        }
        if (lonType.equals("AIT")) {
            proj = new Projection("Ait");
        } else if (lonType.equals("CAR")) {
            proj = new Projection("Car");
        } else if (lonType.equals("CSC")) {
            proj = new Projection("Csc");
        } else if (lonType.equals("SFL") || lonType.equals("GLS")) {
            proj = new Projection("Sfl");
        } else if (lonType.equals("TOA")) {
            proj = new Projection("Toa");
        } else {
            double crval2;
            double crval1 = this.h.getDoubleValue("CRVAL" + this.lonAxis, Double.NaN);
            if (Double.isNaN(crval1 + (crval2 = this.h.getDoubleValue("CRVAL" + this.latAxis, Double.NaN)))) {
                throw new TransformationException("Unable to find reference coordinates in FITS header");
            }
            this.wcsKeys.put("CRVAL1", crval1);
            this.wcsKeys.put("CRVAL2", crval2);
            if (lonType.equals("TAN") || lonType.equals("SIN") || lonType.equals("ZEA") || lonType.equals("ARC") || lonType.equals("STG")) {
                double lonDefault;
                String type = lonType.substring(0, 1) + lonType.substring(1, 3).toLowerCase();
                proj = new Projection(type, new double[]{Math.toRadians(crval1), Math.toRadians(crval2)});
                double lonpole = this.h.getDoubleValue("LONPOLE", Double.NaN);
                if (!Double.isNaN(lonpole)) {
                    this.wcsKeys.put("LONPOLE", lonpole);
                }
                if (!Double.isNaN(lonpole) && lonpole != (lonDefault = 180.0)) {
                    Rotater r = proj.getRotater();
                    Rotater lon = new Rotater("Z", Math.toRadians(lonpole - lonDefault), 0.0, 0.0);
                    if (r != null) {
                        proj.setRotater(r.add(lon));
                    } else {
                        proj.setRotater(lon);
                    }
                }
            } else if (lonType.equals("NCP")) {
                double[] xproj = new double[]{Math.toRadians(crval1), 1.5707963267948966};
                if (crval2 < 0.0) {
                    xproj[1] = -xproj[1];
                }
                double poleOffset = Math.sin(xproj[1] - Math.toRadians(crval2));
                proj = new Projection("Sin", xproj);
                ncpScale = new Scaler(0.0, poleOffset, 1.0, 0.0, 0.0, 1.0);
                ncpScale = ncpScale.add(new Scaler(0.0, 0.0, 1.0, 0.0, 0.0, 1.0 / Math.sin(Math.toRadians(crval2))));
            } else {
                throw new TransformationException("Unsupported projection type:" + lonType);
            }
        }
        this.proj = proj;
        if (ncpScale != null) {
            this.scale = ncpScale;
        }
        this.add(proj.getRotater());
        this.add(proj.getProjecter());
        this.add(ncpScale);
    }

    private boolean checkDSS() throws TransformationException {
        String origin = this.h.getStringValue("ORIGIN");
        if (origin == null || this.h.getStringValue("CTYPE1") != null) {
            return false;
        }
        this.wcsKeys.put("ORIGIN", origin);
        if (this.h.getStringValue("CTYPE1") != null && !preferDSS) {
            return false;
        }
        if (this.h.getDoubleValue("XPIXELSZ", -1.0) == -1.0) {
            return false;
        }
        return origin.startsWith("CASB") || origin.startsWith("STScI");
    }

    private void doDSSWCS() throws TransformationException {
        CoordinateSystem coords;
        Projection proj;
        double plateRA = this.h.getDoubleValue("PLTRAH") + this.h.getDoubleValue("PLTRAM") / 60.0 + this.h.getDoubleValue("PLTRAS") / 3600.0;
        plateRA = Math.toRadians(15.0 * plateRA);
        this.wcsKeys.put("PLTRAH", this.h.getDoubleValue("PLTRAH"));
        this.wcsKeys.put("PLTRAM", this.h.getDoubleValue("PLTRAM"));
        this.wcsKeys.put("PLTRAS", this.h.getDoubleValue("PLTRAS"));
        double plateDec = this.h.getDoubleValue("PLTDECD") + this.h.getDoubleValue("PLTDECM") / 60.0 + this.h.getDoubleValue("PLTDECS") / 3600.0;
        plateDec = Math.toRadians(plateDec);
        if (this.h.getStringValue("PLTDECSN").substring(0, 1).equals("-")) {
            plateDec = -plateDec;
        }
        this.wcsKeys.put("PLTDECD", this.h.getDoubleValue("PLTDECD"));
        this.wcsKeys.put("PLTDECM", this.h.getDoubleValue("PLTDECM"));
        this.wcsKeys.put("PLTDECS", this.h.getDoubleValue("PLTDECS"));
        this.wcsKeys.put("PLTDECSN", this.h.getStringValue("PLTDECSN"));
        double plateScale = this.h.getDoubleValue("PLTSCALE");
        double xPixelSize = this.h.getDoubleValue("XPIXELSZ");
        double yPixelSize = this.h.getDoubleValue("YPIXELSZ");
        this.wcsKeys.put("PLTSCALE", plateScale);
        this.wcsKeys.put("XPIXELSZ", xPixelSize);
        this.wcsKeys.put("YPIXELSZ", yPixelSize);
        double[] xCoeff = new double[20];
        double[] yCoeff = new double[20];
        for (int i = 1; i <= 20; ++i) {
            xCoeff[i - 1] = this.h.getDoubleValue("AMDX" + i);
            yCoeff[i - 1] = this.h.getDoubleValue("AMDY" + i);
            this.wcsKeys.put("AMDX" + i, xCoeff[i - 1]);
            this.wcsKeys.put("AMDY" + i, yCoeff[i - 1]);
        }
        double[] ppo = new double[6];
        for (int i = 1; i <= 6; ++i) {
            ppo[i - 1] = this.h.getDoubleValue("PPO" + i);
            this.wcsKeys.put("PPO" + i, ppo[i - 1]);
        }
        double plateCenterX = ppo[2];
        double plateCenterY = ppo[5];
        double cdelt1 = -plateScale / 1000.0 * xPixelSize / 3600.0;
        double cdelt2 = plateScale / 1000.0 * yPixelSize / 3600.0;
        this.wcsScale = Math.abs(cdelt1);
        double crpix1 = plateCenterX / xPixelSize - this.h.getDoubleValue("CNPIX1", 0.0) - 0.5;
        double crpix2 = plateCenterY / yPixelSize - this.h.getDoubleValue("CNPIX2", 0.0) - 0.5;
        this.wcsKeys.put("CNPIX1", this.h.getDoubleValue("CNPIX1", 0.0));
        this.wcsKeys.put("CNPIX2", this.h.getDoubleValue("CNPIX2", 0.0));
        this.proj = proj = new Projection("Tan", new double[]{plateRA, plateDec});
        this.csys = coords = CoordinateSystem.factory("J2000");
        cdelt1 = Math.toRadians(cdelt1);
        cdelt2 = Math.toRadians(cdelt2);
        Scaler s = new Scaler(-cdelt1 * crpix1, -cdelt2 * crpix2, cdelt1, 0.0, 0.0, cdelt2);
        this.add(coords.getSphereDistorter());
        this.add(coords.getRotater());
        this.add(proj.getRotater());
        this.add(proj.getProjecter());
        this.distort = new DSS(plateRA, plateDec, xPixelSize, yPixelSize, plateScale, ppo, xCoeff, yCoeff);
        this.add(this.distort);
        this.scale = s.inverse();
        this.add(this.scale);
    }

    private boolean checkNeat() {
        return this.h.getStringValue("CTYPE1").equals("RA---XTN");
    }

    private void doNeatWCS() throws TransformationException {
        Projection proj;
        CoordinateSystem csys;
        this.csys = csys = CoordinateSystem.factory("J2000");
        double cv1 = Math.toRadians(this.h.getDoubleValue("RA0"));
        double cv2 = Math.toRadians(this.h.getDoubleValue("DEC0"));
        this.wcsKeys.put("CRVAL1", Math.toDegrees(cv1));
        this.wcsKeys.put("CRVAL2", Math.toDegrees(cv2));
        this.proj = proj = new Projection("Tan", new double[]{cv1, cv2});
        double cd1 = Math.toRadians(this.h.getDoubleValue("CDELT1"));
        double cd2 = Math.toRadians(this.h.getDoubleValue("CDELT2"));
        this.wcsKeys.put("CDELT1", Math.toDegrees(cd1));
        this.wcsKeys.put("CDELT2", Math.toDegrees(cd2));
        this.wcsScale = Math.abs(cd1);
        double cp1 = this.h.getDoubleValue("CRPIX1");
        double cp2 = this.h.getDoubleValue("CRPIX2");
        this.wcsKeys.put("CPRIX1", cp1);
        this.wcsKeys.put("CRPIX2", cp2);
        this.wcsKeys.put("CTYPE1", "RA---XTN");
        this.wcsKeys.put("CTYPE2", "DEC--XTN");
        Scaler s1 = new Scaler(0.0, 0.0, -1.0 / cd1, 0.0, 0.0, -1.0 / cd2);
        double x0 = this.h.getDoubleValue("X0");
        double y0 = this.h.getDoubleValue("Y0");
        Neat dis = new Neat(this.h.getDoubleValue("RADIAL"), this.h.getDoubleValue("XRADIAL"), this.h.getDoubleValue("YRADIAL"));
        double a0 = this.h.getDoubleValue("A0");
        double a1 = this.h.getDoubleValue("A1");
        double a2 = this.h.getDoubleValue("A2");
        double b0 = this.h.getDoubleValue("B0");
        double b1 = this.h.getDoubleValue("B1");
        double b2 = this.h.getDoubleValue("B2");
        double[] cpix = new double[]{cp1, cp2};
        double[] cout = new double[2];
        ((Transformer)dis).transform(cpix, cout);
        Scaler s2 = new Scaler(cout[0] - a0 - a1 * x0 - a2 * y0, cout[1] - b0 - b2 * x0 - b1 * y0, -(1.0 + a1), -a2, -b2, -(1.0 + b1));
        this.wcsKeys.put("A0", a0);
        this.wcsKeys.put("A1", a1);
        this.wcsKeys.put("A2", a2);
        this.wcsKeys.put("B0", b0);
        this.wcsKeys.put("B1", b1);
        this.wcsKeys.put("B2", b2);
        this.distort = dis;
        this.add(csys.getSphereDistorter());
        this.add(csys.getRotater());
        this.add(proj.getRotater());
        this.add(proj.getProjecter());
        this.scale = s1;
        this.scale = this.scale.add(s2.inverse());
        this.add(this.scale);
        this.add(dis);
    }

    public double getScale() {
        return this.wcsScale;
    }

    public void updateHeader(Header h, Scaler s, double[] crval, String projString, String coordString) throws Exception {
        if (this.proj.isFixedProjection()) {
            h.addValue("CRVAL1", this.proj.getReferencePoint()[0], "Fixed reference center");
            h.addValue("CRVAL2", this.proj.getReferencePoint()[1], "Fixed reference center");
        } else {
            h.addValue("CRVAL1", crval[0], "Reference longitude");
            h.addValue("CRVAL2", crval[1], "Reference latitude");
        }
        coordString = coordString.toUpperCase();
        String[] prefixes = new String[2];
        char c = coordString.charAt(0);
        if (c == 'J' || c == 'I') {
            h.addValue("RADESYS", "FK5", "Coordinate system");
            prefixes[0] = "RA--";
            prefixes[1] = "DEC-";
        } else if (c == 'B') {
            h.addValue("RADESYS", "FK4", "Coordinate system");
            prefixes[0] = "RA--";
            prefixes[1] = "DEC-";
        } else {
            prefixes[0] = c + "LON";
            prefixes[1] = c + "LAT";
        }
        if (c != 'G' && c != 'I') {
            try {
                double equinox = Double.parseDouble(coordString.substring(1));
                h.addValue("EQUINOX", equinox, "Epoch of the equinox");
            }
            catch (Exception e) {
                // empty catch block
            }
        }
        if (c == 'I') {
            h.addValue("EQUINOX", 2000L, "ICRS coordinates");
        }
        String upProj = projString.toUpperCase();
        h.addValue("CTYPE1", prefixes[0] + "-" + upProj, "Coordinates -- projection");
        h.addValue("CTYPE2", prefixes[1] + "-" + upProj, "Coordinates -- projection");
        h.addValue("CRPIX1", s.x0 + 0.5, "X reference pixel");
        h.addValue("CRPIX2", s.y0 + 0.5, "Y reference pixel");
        if (Math.abs(s.a01) < 1.0E-14 && Math.abs(s.a10) < 1.0E-14) {
            h.addValue("CDELT1", Math.toDegrees(1.0 / s.a00), "X scale");
            h.addValue("CDELT2", Math.toDegrees(1.0 / s.a11), "Y scale");
        } else {
            Scaler rev = s.inverse();
            h.addValue("CD1_1", Math.toDegrees(rev.a00), "Matrix element");
            h.addValue("CD1_2", Math.toDegrees(rev.a01), "Matrix element");
            h.addValue("CD2_1", Math.toDegrees(rev.a10), "Matrix element");
            h.addValue("CD2_2", Math.toDegrees(rev.a11), "Matrix element");
        }
    }

    public void copyToHeader(Header h) throws HeaderCardException {
        Object[] srt = this.wcsKeys.keySet().toArray(new String[0]);
        Arrays.sort(srt);
        for (Object key : srt) {
            Object o = this.wcsKeys.get(key);
            if (o instanceof Integer) {
                h.addValue((String)key, ((Integer)o).intValue(), "Copied WCS eleemnt");
                continue;
            }
            if (o instanceof Double) {
                h.addValue((String)key, (Double)o, "Copied WCS element");
                continue;
            }
            if (!(o instanceof String)) continue;
            h.addValue((String)key, (String)o, "Copied WCS element");
        }
    }
}

