#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: FocusCalObservation.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase

class FocusCalObservation(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("sourceName", str, ""),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 7),
        Observation.ObsCalBase.scriptOption("axisList", str, "Z")
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None

    def parseOptions(self):
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.userSpecifiedSource     = self.args.sourceName
        self.pointFocusBand          = self.args.pointFocusBand
        axisStr                      = self.args.axisList
        self.axisList = []
        for s in axisStr.split(','):
            if s not in ["X", "Y", "Z"]:
                raise Exception("Invalid axis in axis list: '%s'" % s)
            self.axisList.append(s)
        self.logInfo("Axis list: %s" % str(self.axisList))

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand

    def doPointing(self):
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException, ex:
            print ex
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
                    
    def doFocus(self):
        for axis in self.axisList:
            try:
                focusCal = Observation.FocusCalTarget.FocusCalTarget(
                        SubscanFieldSource = self._srcPointFocus,
                        Axis = axis,
                        SpectralSpec = self._pointFocusSpectralSpec,
                        DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                        SubscanDuration = self.focusSubscanDuration,
                        OneWay = False,
                        NumPositions = 7)
                self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
                focusCal.execute(self._obsmode)
                self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
                result = focusCal.checkResult(self._array)
                self.logInfo("Result is: %s" % str(result))
                if len(result) > 0: 
                    for key in result.keys():
                        self.logInfo("Found solution for %s using polarization(s) %s" %
                                (key, result[key]))
                    focusCal.applyResult(self._obsmode, result)
                else:
                    if not "OSS" in self._array._arrayName:
                        raise Exception("No focus results!")
            except BaseException, ex:
                print ex
                msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex



obs = FocusCalObservation()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource(minEl=30.0, useThisSource=obs.userSpecifiedSource)
except BaseException, ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing first pointing...")
obs.doPointing()
obs.logInfo("Executing second pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing focus...")
obs.doFocus()
obs.logInfo("Executing last pointing -- make sure results are good!...")
obs.doPointing()
obs.closeExecution()

