#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalAzSearch.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global math
import math

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase

global TelCalParameters
import TelCalParameters

class ObsCalAzSearch(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 3.072),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.008),
        Observation.ObsCalBase.scriptOption("excursion", float, 1000.0),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "2 deg"),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 3)
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self, singleDish=True)
        self._srcPointFocus = None

    def parseOptions(self):
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.excursion               = self.args.excursion
        self.elLimit                 = self.args.ElLimit
        self.pointFocusBand          = self.args.pointFocusBand

    def generateTunings(self):
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                frequency = 92.0,
                intent = "total_power",
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing" % self.pointFocusBand

    def setTelCalParams(self):
        self.logInfo("Setting TelCal parameters pointingFitWidth=False, simpleGaussianFit=False")
	tcParameters = self.getTelCalParams()
	tcParameters.setCalibParameter('pointingFitWidth', False)
        tcParameters.setCalibParameter('simpleGaussianFit',False)

    def doPointing(self):
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('TOTAL_POWER')
            pointingCal.setDelayCalReduction(False)
            pointingCal.setPointingMethod('cross')
            pointingCal.setExcursion(math.radians(self.excursion/3600.0))
            pointingCal.specialAzSearch = True
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
            else:
                self.logInfo("Found no solution -- check data offline")
        except BaseException, ex:
            print ex
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex




obs = ObsCalAzSearch()
obs.parseOptions()
#obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource(maxEl=65.0, singleDish=True)
except BaseException, ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.setTelCalParams()
obs.logInfo("Executing azimuth search pointing scan...")
obs.doPointing()
obs.closeExecution()

