#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalIFDelays.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.DelayCalTarget
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalIFDelays(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("band", int, 3),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("bbNames", str, "")
    ]

    def parseOptions(self):
        self.band                    = self.args.band
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        bbNameStr                    = self.args.bbNames
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        fLO1 = Observation.SSRTuning.bandFreqs_delayMeasurement[self.band]
        fLO2 = 9.021e9
        if self.band in [6, 9, 10]:
            fLO2 = 11.021e9
        fSky_LSB = fLO1-fLO2+3.0e9
        fSky_USB = fLO1+fLO2-3.0e9
        #
        # A bit ugly: make a normal SS, then turn it into a hardware one, then
        # make the necessary 4 copies to cover the parameter space.
        #
        origSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.band,
                intent = "interferometry_continuum",
                frequency = fLO1,
                corrType = corrType,
                pol = '4',
                bbNames = self.bbNames,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration)
        origSpec.FrequencySetup.lO1Frequency.set(fLO1)
        origSpec.FrequencySetup.isUserSpecifiedLO1 = True
        origSpec.FrequencySetup.hasHardwareSetup = True
        origSpec.FrequencySetup.floog.set(32.5e6)
        origSpec.FrequencySetup.tuneHigh = True
        origSpec.FrequencySetup.dopplerReference = u'topo'
        for bbs in origSpec.FrequencySetup.BaseBandSpecification:
            bbs.lO2Frequency.set(fLO2)

        self._calSpectralSpecs = []
        for i in range(4):
            self._calSpectralSpecs.append(copy.deepcopy(origSpec))

        # LSB low
        self._calSpectralSpecs[0].name = "band %d LSB low" % self.band
        self._calSpectralSpecs[0].FrequencySetup.restFrequency.set(fSky_LSB)
        for bbs in self._calSpectralSpecs[0].FrequencySetup.BaseBandSpecification:
            bbs.centerFrequency.set(fSky_LSB)
            bbs.useUSB = False
            bbs.use12GHzFilter = False
            bbs.sideBandPreference = unicode('LSB')

        # USB low
        self._calSpectralSpecs[1].name = "band %d USB low" % self.band
        self._calSpectralSpecs[1].FrequencySetup.restFrequency.set(fSky_USB)
        for bbs in self._calSpectralSpecs[1].FrequencySetup.BaseBandSpecification:
            bbs.centerFrequency.set(fSky_USB)
            bbs.useUSB = True
            bbs.use12GHzFilter = False
            bbs.sideBandPreference = unicode('USB')

        # LSB high
        self._calSpectralSpecs[2].name = "band %d LSB high" % self.band
        self._calSpectralSpecs[2].FrequencySetup.restFrequency.set(fSky_LSB)
        for bbs in self._calSpectralSpecs[2].FrequencySetup.BaseBandSpecification:
            bbs.centerFrequency.set(fSky_LSB)
            bbs.useUSB = False
            bbs.use12GHzFilter = True
            bbs.sideBandPreference = unicode('LSB')

        # USB high
        self._calSpectralSpecs[3].name = "band %d USB high" % self.band
        self._calSpectralSpecs[3].FrequencySetup.restFrequency.set(fSky_USB)
        for bbs in self._calSpectralSpecs[3].FrequencySetup.BaseBandSpecification:
            bbs.centerFrequency.set(fSky_USB)
            bbs.useUSB = True
            bbs.use12GHzFilter = True
            bbs.sideBandPreference = unicode('USB')

        for ss in self._calSpectralSpecs:
            self.logInfo("SpectralSpec '%s': %s" % (ss.name, ss.toxml()))

    def doDelayCals(self):
        src = self._srcPointFocus
        for ss in self._calSpectralSpecs:
            try:
                delayCal = Observation.DelayCalTarget.DelayCalTarget(src, ss)
                delayCal.setSubscanDuration(10.0)
                delayCal.setIntegrationTime(1.0)
                self.logInfo('Executing DelayCal on ' + src.sourceName + '...')
                delayCal.execute(self._obsmode)
                self.logInfo('Completed DelayCal on ' + src.sourceName)
            except BaseException, ex:
                print ex
                msg = "Error executing cal survey scans on source %s" % src.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex




obs = ObsCalIFDelays()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource()
except BaseException, ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing DelayCals...")
obs.doDelayCals()
obs.closeExecution()

