#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalSolarFastScan.py 247933 2017-08-08 15:57:43Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global math
import math

global Control
import Control

global CCL
import CCL.APDMSchedBlock

global ControlExceptionsImpl
import ControlExceptionsImpl

global PyDataModelEnumeration
import PyDataModelEnumeration

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase
import Observation.FillPattern
import Observation.FastScanPattern
# import Observation.TargetUtility


class ObsCalSolarFastScan(Observation.ObsCalBase.ObsCalBase):

    pointingExcursion = None
    # These are on-source levels
    ifLevel = -26.0
    bbLevel = 2.0
    mapAttenuatorSettingName = "Map attenuator setting"
    mapSubscanSequenceSpec = None
    mapMixerMode = Control.NORMAL
    _srcPointFocus = None

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self, singleDish=True)

    def parseOptions(self):
        # In ICT-6959 we chose to use the SB representative band instead of expert param
        # Use some canned defaults for solar/moon maps in each band
        repBand = 3
        defMapSpacing = 20.0
        defMapDuration = 290.0
        if self._sb is not None:
            repBandStr = str(self._sb.SchedulingConstraints.representativeReceiverBand)
            repBand = int(repBandStr.split('_')[-1])
            self.logInfo("SB Representative band: %d ('%s')" % (repBand, repBandStr))
        if repBand == 6:
            defMapSpacing = 10.0
            defMapDuration = 570.0
        if repBand == 7:
            defMapSpacing = 8.0
            defMapDuration = 710.0
        if repBand == 9:
            defMapSpacing = 4.0
            defMapDuration = 1406.0
        self.pointingSubscanDuration = self.getExpertParameterOrDefault("pointingSubscanDuration", float, 3.072)
        self.focusSubscanDuration    = self.getExpertParameterOrDefault("focusSubscanDuration", float, 2.0)
        self.atmSubscanDuration      = self.getExpertParameterOrDefault("atmSubscanDuration", float, 4.0)
        self.numFocusPositions       = self.getExpertParameterOrDefault("numFocusPositions", int, 5)
        self.focusOneWay             = self.getExpertParameterOrDefault("focusOneWay", int, 0) > 0
        self.tpIntegrationDuration   = self.getExpertParameterOrDefault("tpIntegrationDuration", float, 0.001)
        # map parameters
        self.mapPatternType          = self.getExpertParameterOrDefault("mapPatternType", str, "CENTERED")
        self.mapPatternSubtype       = self.getExpertParameterOrDefault("mapPatternSubtype", str, "AUTO")
        self.mapWidth                = self.getExpertParameterOrDefault("mapWidth", float, 2400.0)
        self.mapHeight               = self.getExpertParameterOrDefault("mapHeight", float, 2400.0)
        self.mapSpacing              = self.getExpertParameterOrDefault("mapSpacing", float, defMapSpacing)
        self.mapOrientation          = self.getExpertParameterOrDefault("mapOrientation", float, 0.0)
        self.mapMaxFreq              = self.getExpertParameterOrDefault("mapMaxFreq", float, 1.0)
        self.mapDoRef                = self.getExpertParameterOrDefault("mapDoRef", int, 1) > 0
        self.mapRefSubscanDuration   = self.getExpertParameterOrDefault("mapRefSubscanDuration", float, 10.0)
        # FIXME: This should be auto-computed from the pattern and a repeat number or such
        self.mapDuration             = self.getExpertParameterOrDefault("mapDuration", float, defMapDuration)
        # As we sometimes observe the Moon or Sun we are best to default to a long way away from the source!
        self.atmAzOffsetArcsec       = self.getExpertParameterOrDefault("atmAzOffsetArcsec", float, 3600.0)
        self.atmDoZero               = self.getExpertParameterOrDefault("atmDoZero", int, 1) > 0
        self.elLimit                 = self.getExpertParameterOrDefault("ElLimit", str, "2 deg")
        self.pointFocusBand          = self.getExpertParameterOrDefault("band", int, repBand)
        self.sourceName              = self.getExpertParameterOrDefault("source", str, "Sun")
        self.activeSun               = self.getExpertParameterOrDefault("activeSun", int, 0) > 0
        if self.activeSun:
            self.mapMixerMode = Control.MD2
        self.atmAzOffsetRad = math.radians(self.atmAzOffsetArcsec / 3600.0)

    def generateTunings(self):
        frequency = Observation.SSRTuning.bandFreqs_solar[self.pointFocusBand]
        # point/focus tuning will typically have longer sample time than fast scan map
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                # TODO: should be careful here for bands where solar tuning
                # puts a sideband in an atmospheric line
                frequency = frequency,
                intent = "total_power",
                tpSampleTime = 0.016)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        self._mapSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                frequency = frequency,
                intent = "total_power",
                tpSampleTime = self.tpIntegrationDuration)
        self._mapSpectralSpec.name = "Band %d mapping" % self.pointFocusBand


    def setMapSource(self):
        self.mapSrc = self.sourceHelper.getSource(self.sourceName)

    def setTelCalParams(self):
        from Observation.Global  import simulatedArray
        self.logInfo("Setting TelCal parameters pointingFitWidth=False, simpleGaussianFit=False")
        if simulatedArray():
            return
        tcParameters = self.getTelCalParams()
        tcParameters.setCalibParameter('pointingFitWidth', False)
        tcParameters.setCalibParameter('simpleGaussianFit',False)

    def configureScanPattern(self):
        frame = Control.HORIZON
        fp = Observation.FillPattern.FillPattern('%.1f arcsec' % self.mapWidth,
                        '%.1f arcsec' % self.mapHeight,
                        '%.1f arcsec' % self.mapSpacing,
                        '%.3f deg' % self.mapOrientation,
                        Observation.FillPattern.FillPatternType.getType(self.mapPatternType),
                        Observation.FillPattern.FillPatternSubtype.getSubtype(self.mapPatternSubtype))
        fsp = Observation.FastScanPattern.FastScanPattern()
        fsp.limitFrequencyMax = self.mapMaxFreq
        spec = fsp.generatePatternSpec(fp)
        self.logInfo("Map pattern spec: " + str(spec._PatternSpec__idlSpec))
        self.mapOffsetSpec = Control.SourceOffset(longitudeOffset=0.0, latitudeOffset=0.0,
                                  longitudeVelocity=0.0, latitudeVelocity=0.0,
                                  pattern = spec._PatternSpec__idlSpec,
                                  frame=frame)


    # This should probably move to PointingCalTarget
    def setPointingExcursion(self):
        clight = 299792458.0
        antennaDiameter = 12.0
        # doing this here where we can't account for pointing subarrays is a bit dumb
        for ant in self._array.antennas():
                if ant[0] == 'C':
                        antennaDiameter = 7.0
                        self.logInfo("Array contains at least one CM antenna, so using D=7m for beam sizes.")
                        break
        obsFreq = self._pointFocusSpectralSpec.getMeanFrequency()
        sourceDiameter = 0.0
        # Remove TargetUtility (ICT-7795)
        # try:
        #     sourceDiameter = Observation.TargetUtility.getAngularDiameter(self._srcPointFocus)
        #     if sourceDiameter is None: sourceDiameter = 0.0
        # except: pass
        # self.logInfo("Assumed source diameter: %.2f arcsec" % (3600.0*math.degrees(sourceDiameter)))
        try:
            from Observation.CalibratorSource import SSOCalibratorSource
            source = SSOCalibratorSource(self._srcPointFocus.sourceName)
            sourceDiameter = source.getAngularDiameter() / (3600. * 180. / math.pi)
        except:
            pass
        self.logInfo("Assumed source diameter: %.2f arcsec" % (sourceDiameter))
        beam = 1.22 * (clight / obsFreq) / antennaDiameter
        excursion = math.sqrt(4.0*beam*beam + sourceDiameter*sourceDiameter)
        self.logInfo("Assumed antenna diamter: %.1f m" % (antennaDiameter))
        self.logInfo("Assumed beam FWHM:       %.2f arcsec" % (beam * (3600.0 * 180.0 / math.pi)))
        self.logInfo("Chosen excursion:        %.2f arcsec" % (excursion * (3600.0 * 180.0 / math.pi)))
        self.pointingExcursion = excursion


    def doPointing(self):
        if self.pointingExcursion is None:
            self.setPointingExcursion()
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('TOTAL_POWER')
            pointingCal.setDelayCalReduction(False)
            pointingCal.setPointingMethod('cross')
            pointingCal.setExcursion(self.pointingExcursion)
            pointingCal.setWVRCalReduction(False)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException, ex:
            print ex
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex


    def doFocus(self):
        ref = CCL.APDMSchedBlock.Reference()
        ref.setCoordinates('-600 arcsec',0,u'horizon')
        ref.cycleTime.set(1.0)
        ref.subScanDuration.set(self.focusSubscanDuration)
        ref.integrationTime.set(self.focusSubscanDuration)
        self._srcPointFocus.Reference = [ref]
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            focusCal.setSubscanDuration(self.focusSubscanDuration)
            focusCal.setIntegrationTime(3.0 * self.focusSubscanDuration)
            focusCal.setDataOrigin('TOTAL_POWER')
            focusCal.setNumPositions(self.numFocusPositions)
            focusCal.setOneWayScan(self.focusOneWay)
            focusCal.setWVRCalReduction(False)
            self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
            # ICT-9503: avoid reference getting used by pointing or any other scans by accident.
            self._srcPointFocus.Reference = []
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                             (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No focus results!")
        except BaseException, ex:
            print ex
            msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex


    def doAtmCal(self):
        try:
            atm = Observation.AtmCalTarget.AtmCalTarget(self.mapSrc, self._mapSpectralSpec, doHotLoad=True)
            atm.setOnlineProcessing(True)
            atm.setDoZero(self.atmDoZero)
            atm.setSubscanDuration(self.atmSubscanDuration)
            atm.setIntegrationTime(1.5)
            atm.setMixerMode(self.mapMixerMode)
            if self._mapSpectralSpec.hasCorrelatorConfiguration():
                    atm.setDataOrigin('FULL_RESOLUTION_AUTO')
                    atm.setWVRCalReduction(True)
            else:
                    atm.setDataOrigin('TOTAL_POWER')
                    atm.setWVRCalReduction(False)
            atm.setApplyWVR(False)
            # Currently we need to set a reference source to use an offset.
            atm._referenceSource=atm._source
            atm._referenceOffset=CCL.SourceOffset.stroke(self.atmAzOffsetRad,0,0,0,Control.HORIZON)
            self.logInfo('Executing AtmCal on ' + self.mapSrc.sourceName + '...')
            atm.execute(self._obsmode)
            self.logInfo('Completed AtmCal on ' + self.mapSrc.sourceName)
        except BaseException, ex:
            print ex
            msg = "Error executing AtmCal on source %s" % self.mapSrc.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex


    def prepareMapSubscanSequenceSpec(self):
            # Prepare subscan sequence specification
            ssl = CCL.SubscanList.SubscanList()
            subscanDuration = 0.048*(1 + int(self.mapDuration / 0.048))
            ssl.addSubscan(self.mapSrc, self._mapSpectralSpec, subscanDuration,
                            SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.ON_SOURCE,
                            CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                            PointingOffsetSpec    = self.mapOffsetSpec,
                            attenuatorSettingName = self.mapAttenuatorSettingName,
                            MixerMode             = self.mapMixerMode)
            if self.mapDoRef:
                    referenceOffset = CCL.SourceOffset.stroke(self.atmAzOffsetRad, 0, 0, 0, Control.HORIZON)
                    self.logInfo("referenceOffset = %s" % str(referenceOffset))
                    subscanDuration = 0.048*(1 + int(self.mapRefSubscanDuration / 0.048))
                    ssl.addSubscan(self.mapSrc, self._mapSpectralSpec, subscanDuration,
                                    SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.OFF_SOURCE,
                                    CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                                    PointingOffsetSpec    = referenceOffset,
                                    attenuatorSettingName = self.mapAttenuatorSettingName,
                                    MixerMode             = self.mapMixerMode)
            subscanSpec = ssl.getSubscanSequenceSpecification()
            self.logInfo('subscanSpec.subscans = %s' % str(subscanSpec.subscans))
            self.mapSubscanSequenceSpec = subscanSpec

    def enableFocusModels(self, enable, continueOnError=False, noExcept=False):
        enableStr = "Enabling" if enable else "Disabling"
        exToRaise = None
        self.logInfo(enableStr + " focus models...")
        for ant in self._array.antennas():
            self.logInfo(enableStr + " focus model for antenna %s" % ant)
            try:
                 if not "OSS" in self.arrayName:
                     CCL.Mount.Mount(ant).enableFocusModel(enable)
            except BaseException, ex:
                print ex
                msg = "Error %s focus model for antenna %s" % (enableStr, ant)
                self.logError(msg)
                if continueOnError:
                    exToRaise = ex
                else:
                    raise ex
        if exToRaise is not None and not noExcept:
            raise exToRaise

    def executeMapScan(self):
            self.logInfo("Parking ACD so we optimise attenuators on-sky...")
            try:
                self._obsmode.setCalibrationDevice(PyDataModelEnumeration.PyCalibrationDevice.NONE)
            except BaseException, ex:
                self.logException('Exception thrown by obsmode.setCalibrationDevice(), considering this fatal!', ex)
                raise ex
            # Just before execution turn off the focus model when pointed at the source
            self.logInfo("Slewing to source with obsmode.setPointingDirection() to set focus before disabling focus models...")
            try:
                self._obsmode.setPointingDirection(self.mapSrc)
            except BaseException, ex:
                self.logException('Exception thrown by obsmode.setPointingDirection(), considering this fatal!', ex)
                raise ex
            try:
                self.enableFocusModels(enable=False, continueOnError=False, noExcept=False)
            except BaseException, ex:
                self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
                raise ex
            # the obvserving mode seems to set the levels after starting the motion pattern, so we have to do it ourselves and save the result. Ugh.
            self.logInfo("Optimising attenuators and saving to setting named '%s'" % self.mapAttenuatorSettingName)
            try:
                loMode = self._obsmode.getLOObservingMode()
                loMode.optimizeSignalLevels(ifTargetLevel = self.ifLevel, bbTargetLevel = self.bbLevel, settingName = self.mapAttenuatorSettingName)
            except BaseException, ex:
                self.logException('Exception thrown by optimizeSignalLevels() or getting reference to LOObservingMode, considering this fatal!', ex)
                raise ex
            # Sort out correlator calibration IDs
            if self.mapSubscanSequenceSpec.spectra[0].hasCorrelatorConfiguration():
                self.logInfo('Performing correlator calibration via getCalibrationIds() with validity interval %d seconds'
                                    % self._obsmode.getCorrelatorCalibrationValidityInterval())
                try:
                    obsmode.getCalibrationIds(self.mapSubscanSequenceSpec)
                except Exception, ex:
                    self.logException('Exception thrown by obsmode.getCalibrationIds(), considering this fatal!', ex)
                    raise ex
            else:
                self.logInfo('No correlator configuration, so just setting calibration IDs to -1 to save effort calling getCalibrationIds()')
                for s in self.mapSubscanSequenceSpec.subscans:
                    s.calibrationId = -1
            # Execute the sequence
            self.logInfo("Now executing the subscan sequence...")
            try:
                self._obsmode.doSubscanSequence(self.mapSubscanSequenceSpec)
            except Exception, ex:
                self.logException('Exception thrown by obsmode.doSubscanSequence(), considering this fatal!', ex)
                raise ex
            self.logInfo('Subscan sequence finished')
            # Re-enable focus models, allowing an exception if any antennas fail
            try:
                self.enableFocusModels(enable=True, continueOnError=True, noExcept=False)
            except BaseException, ex:
                self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
                raise ex

    def doMap(self):
            self.prepareMapSubscanSequenceSpec()
            scanIntent = CCL.ScanIntent.ScanIntent(PyDataModelEnumeration.PyScanIntent.OBSERVE_TARGET)
            self.logInfo('Now beginning the scan')
            self._obsmode.beginScan(scanIntent)
            try:
                self.executeMapScan()
            except BaseException, ex:
                self.logError("Error executing map scan, cleaning up and exiting")
                self.enableFocusModels(enable=True, continueOnError=True, noExcept=True)
                self._obsmode.endScan()
                self.closeExecution(ex)
                raise ex
            self._obsmode.endScan()
            self.logInfo("Scan ended")


Observation.ObsCalBase.checkArray()

obs = ObsCalSolarFastScan()
obs.parseOptions()
#obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.setMapSource()
    # TODO: pass source name parameter sunMoon here so can look for a close source
    obs.findPointFocusSource(minEl=25.0, maxEl=80.0, singleDish=True)
    obs.configureScanPattern()
except BaseException, ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.setTelCalParams()
obs.logInfo("Executing first pointing...")
obs.doPointing()
obs.logInfo("Executing second pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing focus...")
obs.doFocus()
obs.logInfo("Executing last pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing AtmCal...")
obs.doAtmCal()
obs.logInfo("Executing Map...")
obs.doMap()
obs.logInfo("Executing AtmCal...")
obs.doAtmCal()
obs.closeExecution()

