#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalSourceCheck.py 247920 2017-08-08 15:07:12Z ahirota $"

# Based on ObsCalGrids.py. Things in common with ObsCalGrids should be 
# moved to a centralized place afterward.

#
# forcing global imports is due to an OSS problem
#
global sys
import sys
global math
import math
global datetime
import datetime
global re
import re
global os
import os
global traceback
import traceback


global CCL
import CCL.Global
import CCL.FieldSource

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global AcsutilPy
import AcsutilPy.FindFile

global Observation
import Observation.AtmCalTarget
import Observation.AmplitudeCalTarget
import Observation.BandpassCalTarget
import Observation.CheckSourceCalTarget
import Observation.PhaseCalTarget
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase
import Observation.ObsCalSources
import Observation.ShadowingChecker
import Observation.CalibratorCatalog


class ObsCalSourceCheck(Observation.ObsCalBase.ObsCalBase):

    # azoffset = math.radians(150.0/3600.0)

    options = [
        # Observation.ObsCalBase.scriptOption("RepeatCount", int, 1),
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("maxShadowFraction", float, 0.3),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 0),
        Observation.ObsCalBase.scriptOption("pipelineFriendly", bool, True),
        Observation.ObsCalBase.scriptOption("bandList", str, "7"),
        Observation.ObsCalBase.scriptOption("scandurationList", str, ""),
        # # Following two parameters are only relevant for pipeline mode.
        # Observation.ObsCalBase.scriptOption("allowHeterogeneousArray", bool, False),
        # Observation.ObsCalBase.scriptOption("allowMultipleBands", bool, False),
        Observation.ObsCalBase.scriptOption("sourceListFile", str, ""),
        Observation.ObsCalBase.scriptOption("extraSourceListFile", str, ""),
        Observation.ObsCalBase.scriptOption("logFile", str, ""),
        Observation.ObsCalBase.scriptOption("noFluxCal", bool, False),
        Observation.ObsCalBase.scriptOption("siteLatitude", float, -23.0231944),
        Observation.ObsCalBase.scriptOption("sources", str, ""),
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self._reverseSpecs = False
        antennas = self._array.antennas()
        self._shadowingChecker = Observation.ShadowingChecker.ShadowingChecker(useSSRLogger=True,
                                                                               usedAntennaList=antennas)
        self.groupLSTBoundaries = [
            [1.75, 3.50],
            [5.50, 8.50],
            [10.00, 12.00],
            [14.75, 16.75],
            [17.75, 18.50],
            [19.75, 23.00],
        ]

        self.overlapSourceList = [
            ["J2258-2758", "J0635-7516"],
            ["J0423-0120", "J0854+2006"],
            ["J0635-7516", "J1256-0547"],
            ["J1229+0203", "J1642+3948"],
            ["J1550+0527", "J2025+3343", "J2148+0657"],
            ["J1924-2914", "J2357-5311"],
        ]

    # Same as ObsCalGrid.py
    def pipelineCheck(self):
        self.logInfo("Checking setup is compatible with PIPELINE requirements...")
        ants = self._array.antennas()
        errStr = ""
        num7m = 0
        num12m = 0
        if len(ants) < 2:
            errStr += " Array has less than two antennas."
        for a in ants:
            if a.startswith("CM"):
               num7m += 1
            else:
               num12m += 1
        if num7m > 0 and num12m > 0:
            # and not self.allowHeterogeneousArray:
            errStr += " Array contains both 7m and 12m antennas."
        if len(self.bandList) > 1:
            # and not self.allowMultipleBands:
            errStr += " Observing band list contains more than one band (%s)." % str(self.bandList)
        if len(errStr) > 0:
            msg = "Pipeline-friendly mode requires all antnenas to be the same size, and only one observing band. Aborting due to:%s" % errStr
            ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            raise ex
        self.logInfo("Setup is compatible with PIPELINE requirements")

    def parseOptions(self):
        # self.repeatCount             = self.args.RepeatCount
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.atmSubscanDuration      = self.args.AtmSubscanDuration
        self.sbrSubscanDuration      = self.args.SBRSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        # self.numTargets              = self.args.NumTargets
        self.pointFocusBand          = self.args.pointFocusBand
        self.pipelineFriendly        = self.args.pipelineFriendly
        bandStr                      = self.args.bandList
        self.sourceListFile          = self.args.sourceListFile
        self.logFile                 = self.args.logFile
        # FIXME: Not implemented yet
        self.extraSourceListFile     = self.args.extraSourceListFile
        self.maxShadowFraction       = self.args.maxShadowFraction
        self.noFluxCal               = self.args.noFluxCal
        self.siteLatitude            = self.args.siteLatitude
        self.sources                 = self.args.sources

        self.bandList = []
        for s in bandStr.split(','):
            n = int(s)
            if n < 1 or n > 10:
                raise Exception("Invalid band number in band list: '%s'" % s)
            self.bandList.append(n)

        if self.pointFocusBand == 0:
            if len(self.bandList) == 0:
                raise Exception("No observin band has been specified")
            self.pointFocusBand = self.bandList[0]
        self.logInfo("Band list: %s" % str(self.bandList))
        if self.pipelineFriendly:
            self.pipelineCheck()

    # Same as ObsCalGrid.py
    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        self._calSpectralSpecs = []
        for band in self.bandList:
            ss = self._tuningHelper.GenerateSpectralSpec(
                            band = band,
                            intent = "calsurvey",
                            corrType = corrType,
                            dualMode = True,
                            dump = self.dumpDuration,
                            channelAverage = self.channelAverageDuration,
                            integration = self.integrationDuration,
                            tpSampleTime = self.tpIntegrationDuration)
            ss.name = "Band %d calsurvey" % band
            self._calSpectralSpecs.append(ss)

    def pickupAvailableSSOs(self):
        """
        ICT-7107: pipeline friendly way of selecting SSO
        """
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing')
        # ICT-9098: use alternative priority ordering of SSO list.
        cc.setGridSurveySSOPriorityOrder()
        sources = None
        for ss in self._calSpectralSpecs:
            # Calculate flux threshold
            minFluxList = cc.getRequiredCalibratorFlux("amplitude", el=40.,
                                                       spectralSpec=ss)
            minFlux = max(minFluxList)

            # Get available SSOs for this spectral spec
            wavelength = 299792458. / ss.getMeanFrequency()
            specSources = cc.getAvailableSSOs(wavelength,
                                              minEl=40.,
                                              maxEl=85.,
                                              duration=1200.,
                                              nRequest=-1,
                                              checkSourceNumbers=False,
                                              maxShadowFraction=0.2,
                                              fluxThreshold=minFlux,
                                              verbose=False)

            srcNameStr = ",".join([source.getName() for source in specSources])
            values = (len(specSources), srcNameStr, ss.name)
            msg = "Picked up %d sources (%s) for spectral spec '%s'" % values
            self.logInfo(msg)

            # Take intersection of sources
            if sources is None:
                sources = specSources
            else:
                sources = [source for source in sources if source in specSources]
        srcList = [source.buildFieldSource() for source in sources]
        return srcList

    def getLSTAndCurrentGroup(self):
        import numpy as np
        # Get current LST and identify group in which we are now.
        lst = self.sourceHelper.getLSTsecondOfDay() / 3600.
        groupMidLSTList = np.mean(self.groupLSTBoundaries, axis=1)
        diffLST = groupMidLSTList - lst
        diffLST[diffLST < -12.] += 24.
        diffLST[diffLST > 12.] -= 24.
        iGroup = np.argmin(np.abs(diffLST))
        return lst, iGroup + 1

    def pickupGridFluxCalibrator(self, grids):
        """
        ICT-8459: pick up an 'overlap' source, when no SSO is available
        as a primary flux calibrator.
        """
        import numpy as np
        # Identify the current group.
        lst, groupNumber = self.getLSTAndCurrentGroup()
        msg = "LST=%.2f [h]: Closest group=%d" % (lst, groupNumber)
        self.logInfo("[pickupGridFluxCalibrator] %s" % msg)

        iGroup = groupNumber - 1
        # Get FieldSource instance for overlap sources in the current group.
        candidates = []
        for sourceName in self.overlapSourceList[iGroup]:
            srcs = [g_ for g_ in grids if g_.sourceName == sourceName]
            if len(srcs) != 1:
                msg = "Cannot get source instance for '%s'" % sourceName
                self.logWarning(msg)
                continue
            candidates.append(srcs[0])
        if len(candidates) == 0:
            msg = "Cannot get any source instance for following sources"
            msg += " [%s]" % ", ".join(self.overlapSourceList[iGroup])
            ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            raise ex

        # Compare elevation of the overlap sources and take the highest one.
        elList = []
        for src in candidates:
            az, el = map(np.degrees, self.sourceHelper.sourceAzEl(src))
            msg = "[%s] az=%6.1f el=%6.1f" % (src.sourceName, az, el)
            self.logInfo("[pickupGridFluxCalibrator]   %s" % msg)
            elList.append(el)
        iCandidate = np.argmax(elList)
        src = candidates[iCandidate]

        msg = "Picked up '%s'" % (src.sourceName)
        self.logInfo("[pickupGridFluxCalibrator] %s" % msg)
        return src

    def getIllegalParameterError(self, msg):
        ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
        ex.setData(Control.EX_USER_ERROR_MSG, msg)
        return ex

    def _readFile(self, fPath):
        if not os.path.isfile(fPath):
            msg = "Unable to find the specified file '%s'" % (fPath)
            raise self.getIllegalParameterError(msg)
        with file(fPath) as f:
            lines = [line.strip() for line in f.readlines()]
        return lines

    def readObservingLog(self):
        import collections
        observedCounts = collections.defaultdict(int)
        if self.logFile == "":
            msg = "Log file for recording source executions not specified"
            self.logWarning(msg)
            return observedCounts

        pat_sep = re.compile("[ \t]+")
        if not os.path.isfile(self.logFile):
            with file(self.logFile, "w") as f:
                pass
        lines = self._readFile(self.logFile)
        for iLine, line in enumerate(lines, 1):
            if len(line) == 0 or line.startswith("#"):
                continue
            tokens = pat_sep.split(line)
            if len(tokens) != 2:
                msg = "[%s:L%d] Illegal line in the log file ('%s')" % (self.logFile, iLine, line)
                raise self.getIllegalParameterError(msg)
            uid = tokens[0]
            srcName = tokens[1]
            if not Observation.Global.simulatedArray() and uid == 'uid://A00/X00/X00':
                # This is a dummy UID used by OSS simulations.
                continue
            observedCounts[srcName] += 1
        return observedCounts

    def readSources(self):
        pat_sep = re.compile("[ \t]+")
        sourceDict = dict()
        if self.sources != "":
            # Users can specify fixed list of source with 'sources' expert
            # parameter.
            srcNames = self.sources.split(",")
            for srcName in srcNames:
                src = self.sourceHelper.getSource(srcName, onlyALMA=False)
                if src is None:
                    msg = "Specified source ('%s') could not be resolved" % srcName
                    raise self.getIllegalParameterError(msg)
                sourceDict[srcName] = dict(fs=src)
            return sourceDict

        if self.sourceListFile == "":
            msg = "Both 'sources' and 'sourceListFile' options are not specified"
            raise self.getIllegalParameterError(msg)

        observedCounts = self.readObservingLog()
        lines = self._readFile(self.sourceListFile)
        for iLine, line in enumerate(lines, 1):
            if len(line) == 0 or line.startswith("#"):
                continue
            tokens = pat_sep.split(line)
            if len(tokens) != 3:
                msg = "[%s:L%d] Unexpected format ('%s')" % (self.sourceListFile, iLine, line)
                raise self.getIllegalParameterError(msg)
            srcName = tokens[0]
            priority = int(tokens[1])
            execCount = int(tokens[2])

            if srcName in sourceDict:
                msg = "[%s:L%d] A duplicated entry found for '%s'" % (self.sourceListFile, iLine, srcName)
                raise self.getIllegalParameterError(msg)

            # Check whether enough number of observations were made for this source
            if observedCounts[srcName] >= execCount:
                msg = "[%s] Already reached required execution count" % srcName
                self.logInfo(msg)
                continue

            ra, dec = self.getApproximateCoordinates(srcName)
            values = (srcName, ra, dec, observedCounts[srcName], execCount)
            msg = "%s ra=%9.4f dec=%9.4f execCount=(%2d/%2d)" % values
            self.logInfo("[readSources] %s" % msg)

            sourceDict[srcName] = dict(priority=priority,
                                       # execCount=execCount,
                                       approxRa=ra,
                                       approxDec=dec,
                                       fs=None)
        return sourceDict

    def getApproximateCoordinates(self, srcName):
        """
        Parse the source name and obtain approxmiate source coordinates.
        """
        pat_qso_name = re.compile("J(\d{2})(\d{2})([+-]\d{2})(\d{2})")
        m = pat_qso_name.findall(srcName)
        if not (len(m) == 1 and len(m[0]) == 4):
            msg = "Source name should be specified in the form 'Jhhmm[+-]ddmm'"
            msg += " ('%s' specified)" % (srcName)
            raise self.getIllegalParameterError(msg)
        m = m[0]
        ra = (int(m[0]) + int(m[1]) / 60.) * 15.
        if m[2].startswith("-"):
            dec = (int(m[2]) - int(m[3]) / 60.)
        else:
            dec = (int(m[2]) + int(m[3]) / 60.)
        return ra, dec

    def pickupCalSources(self):
        self.sourceDict = self.readSources()
        # Estimate source flux for pointing/focus source for cal spectralspec
        # SPW center frequencies.
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing')
        cc.estimateSourceFlux(self._srcPointFocus, self._calSpectralSpecs,
                              setSourceProperties=True)
        self.bandpassCalSource = self._srcPointFocus

        if self.noFluxCal:
            self.ampCalSources = []
            return
        # For the convenience, we will populate source flux properties for
        # grid sources. To do this, we have to pass the list of spectral
        # setups to getAllGridSources() so that it knows for which frequencies
        # flux estimation has to be made. (Note that getAllGridSources() and
        # Observation.CalibratorSource.getFluxEstimate() were updated for this.
        grids = self.sourceHelper.getAllGridSources(self._calSpectralSpecs)

        # Pick up primary flux calibrator(s)
        ssoSources = self.pickupAvailableSSOs()

        if len(ssoSources) > 0:
            self.ampCalSources = list(ssoSources)
        else:
            # No SSO available: will use one of overlap source as a
            # primary flux calibrator.
            self.ampCalSources = [self.pickupGridFluxCalibrator(grids)]

        reservedSources = [self.bandpassCalSource.sourceName]
        for src in self.ampCalSources:
            reservedSources.append(src.sourceName)

        # TODO: From CYCLE-6, "fluxcal" is available
        srcList = cc.getBrightSources("grid",
                                      minEl=30.,
                                      maxEl=82.,
                                      nRequest=-1,
                                      duration=1800.,
                                      returnFieldSource=True)

        self.secondaryAmpCalSources = []
        for src in srcList:
            sName = src.sourceName
            if sName in reservedSources:
                continue
            self.logInfo("Use '%s' as a secondary flux calibrator" % (sName))
            self.secondaryAmpCalSources = [src]
            break

    # Same as ObsCalGrid.py
    def orderedSpecs(self):
        ret = self._calSpectralSpecs
        if self._reverseSpecs:
            ret = reversed(self._calSpectralSpecs)
        self._reverseSpecs = not self._reverseSpecs
        return ret

    # Same as ObsCalGrid.py
    def doPointing(self):
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            if not self.pipelineFriendly:
                pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException, ex:
            print ex
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    # Same as ObsCalGrid.py
    def doFocus(self):
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(
                    SubscanFieldSource = self._srcPointFocus,
                    Axis = 'Z',
                    SpectralSpec = self._pointFocusSpectralSpec,
                    DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                    SubscanDuration = self.focusSubscanDuration,
                    OneWay = False,
                    NumPositions = 7)
            self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in result.keys():
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No focus results!")
        except BaseException, ex:
            print ex
            msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def doATMCal(self, src, ss):
        atm = Observation.AtmCalTarget.AtmCalTarget(src, ss, doHotLoad=True)
        atm.setOnlineProcessing(True)
        atm.setDataOrigin('FULL_RESOLUTION_AUTO')
        atm.setDoZero(False)
        atm.setSubscanDuration(self.atmSubscanDuration)
        atm.setIntegrationTime(1.5)
        atm.setWVRCalReduction(True)
        # Applying the results takes a while with lots of
        # antennas, so until we use online WVR, don't bother
        atm.setApplyWVR(False)

        # Automatically adjust reference position for this target.
        atm.tweakReferenceOffset()
        self.logInfo('Executing AtmCal on ' + src.sourceName + '...')
        atm.execute(self._obsmode)
        self.logInfo('Completed AtmCal on ' + src.sourceName)

    def determineSubscanDuration(self, ss, intent):
        # TODO: Handle for weaker sources
        subscanDuration = 0.3e-9 * ss.getMeanFrequency()
        return subscanDuration

    def doCalSource(self, src, intent):
        for ss in self.orderedSpecs():
            try:
                self.doATMCal(src, ss)
                subscanDuration = self.determineSubscanDuration(ss, intent)
                if intent == 'bandpass':
                    target = Observation.BandpassCalTarget.BandpassCalTarget(src, ss)
                elif intent == 'flux':
                    target = Observation.AmplitudeCalTarget.AmplitudeCalTarget(src, ss)
                elif intent == 'check':
                    target = Observation.CheckSourceCalTarget.CheckSourceCalTarget(src, ss)
                elif intent == 'phase':
                    target = Observation.PhaseCalTarget.PhaseCalTarget(src, ss)
                else:
                    msg = "Unsupported intent specified '%s'" % (intent)
                    raise self.getIllegalParameterError(msg)

                target.setSubscanDuration(subscanDuration)
                target.setIntegrationTime(1.0)
                sName = src.sourceName
                self.logInfo('Executing %sCal on %s ...' % (intent.capitalize(), sName))
                target.execute(self._obsmode)
                self.logInfo('Completed %sCal on %s' %  (intent.capitalize(), sName))

                # band = ss.FrequencySetup.receiverBand.replace("ALMA_RB_", "")
            except BaseException, ex:
                self.logError(traceback.format_exc())
                msg = "Error executing cal survey scans on source %s" % src.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex

    @staticmethod
    def calcAzEl(raRad, decRad, lstHour, siteLatitudeRad):
        # TODO: should be moved to a more appropriate place
        import numpy as np
        lst = np.radians((360.0/24.0) * lstHour)
        cosDec = np.cos(decRad)
        sinDec = np.sin(decRad)
        cosLat = np.cos(siteLatitudeRad)
        sinLat = np.sin(siteLatitudeRad)
        cosHa = np.cos(lst - raRad)
        sinHa = np.sin(lst - raRad)
        az = np.arctan2(-cosDec * sinHa,
                        cosLat * sinDec - sinLat * cosDec * cosHa)
        az = az % (2.0 * np.pi)
        el = np.arcsin(sinLat * sinDec + cosLat * cosHa * cosDec)
        return az, el

    @staticmethod
    def calcHAOffsetFromElevation(decRad, elRad, siteLatRad):
        # TODO: should be moved to a more appropriate place
        import numpy as np
        sinLat_ = np.sin(siteLatRad)
        cosLat_ = np.cos(siteLatRad)
        sinDec_ = np.sin(decRad)
        cosDec_ = np.cos(decRad)
        cosHA = (np.sin(elRad) - sinDec_ * sinLat_) / (cosLat_ * cosDec_)
        if np.abs(cosHA) > 1:
            return np.nan
        absHa = np.arccos(cosHA)
        return absHa

    def selectNextSources(self, N=3,
                          minEl=20., maxEl=80.,
                          preferredElMin=50.):
        import numpy as np
        srcNames = self.sourceDict.keys()
        lstHour = self.sourceHelper.getLSTHour()
        self.logInfo("LST = %.2f [hour]" % (lstHour))

        siteLatitudeRad = np.radians(self.siteLatitude)
        candidates = []
        for srcName in srcNames:
            aRa = self.sourceDict[srcName]["approxRa"]
            aDec = self.sourceDict[srcName]["approxDec"]
            aAz, aEl = self.calcAzEl(np.radians(aRa), np.radians(aDec),
                                     lstHour, siteLatitudeRad)
            aAz = np.degrees(aAz)
            aEl = np.degrees(aEl)
            if aEl < minEl or aEl > maxEl:
                continue
            fShadow = self._shadowingChecker.isBlocked(aAz, aEl,
                                                       returnFraction=True)
            aElMax = 90. - np.abs(aDec - self.siteLatitude)

            # Derive how long the source will be over the preferred lower
            # limit of the elevation. This one could be 'nan'.
            haOff = self.calcHAOffsetFromElevation(np.radians(aDec),
                                                   np.radians(preferredElMin),
                                                   siteLatitudeRad)
            timeToBeCaught = (aRa / 15. + haOff * (12. / np.pi)) - lstHour
            timeToBeCaught = timeToBeCaught % 24.
            if timeToBeCaught > 12:
                timeToBeCaught -= 24.

            # Derive how long the source will be over 20 degrees
            # This one also could be 'nan' .
            haOff = self.calcHAOffsetFromElevation(np.radians(aDec),
                                                   np.radians(20.),
                                                   siteLatitudeRad)
            durationOverElLimit = haOff * (12. / np.pi) * 2.

            score = 5.
            if fShadow > 0.:
                score = score / (fShadow / 0.03)
            if np.isnan(durationOverElLimit):
                score = -1.
            elif durationOverElLimit > 1.:
                score = score / durationOverElLimit ** 0.5
            if timeToBeCaught > -0.2:
                score = score / np.maximum(timeToBeCaught, 0.5)
            if aEl < preferredElMin:
                score = score * aEl / preferredElMin
            if aEl / aElMax < 0.4:
                score = -1.
            if fShadow > self.maxShadowFraction:
                score = -1.
            entry = [srcName, aAz, aEl, fShadow, durationOverElLimit, timeToBeCaught, score]
            candidates.append(entry)

        sortedCandidates = sorted(candidates, key=lambda c: c[-1], reverse=True)
        for srcName, aAz, aEl, fShadow, durationOverElLimit, timeToBeCaught, score in sortedCandidates:
            values = (srcName, aAz, aEl, fShadow, 20., durationOverElLimit, timeToBeCaught, score)
            msg = "[%s] (approx) (az,el)=(%3d,%3d) fShadow=%.1f d(El>%.1f)=%5.1f [hour]  remain=%4.1f [hour]  score=%4.2f" % values
            self.logInfo(msg)
        selectedSrcNames = [c[0] for c in sortedCandidates[:N] if c[-1] > 0]
        # Obtain FieldSource instances
        for srcName in selectedSrcNames:
            if self.sourceDict[srcName]["fs"]:
                continue
            src = self.sourceHelper.getSource(srcName, onlyALMA=False)
            if src is None:
                msg = "Could not resolve '%s'" % (srcName)
                self.logWarning(msg)
            self.sourceDict[srcName]["fs"] = src
        # Ignore sources which could not be resolved.
        selectedSrcNames = [sN for sN in selectedSrcNames if self.sourceDict[sN]["fs"]]
        return selectedSrcNames

    def writeLogEntry(self, srcName):
        if self.logFile == "":
            return
        self.logInfo("writeLogEntry(): adding record for '%s'" % srcName)
        oldUmask = os.umask(0)
        fd = os.open(os.path.expanduser(self.logFile), os.O_WRONLY | os.O_CREAT | os.O_APPEND, 0666)
        fp = os.fdopen(fd, "a")
        fp.write("%-21s %s\n" % (str(self.uid), str(srcName)))
        fp.close()
        os.umask(oldUmask)

    def determinePreferredMinimumElevation(self):
        antennas = self._array.antennas()
        nCM = len([ant for ant in antennas if ant.startswith("CM")])
        if nCM > 6:
            preferredElMin = 50.
        else:
            preferredElMin = 40.
        msg = "%d CM antennas included in the array." % (nCM)
        msg += " Will Prefer sources over el=%d deg" % (preferredElMin)
        self.logInfo(msg)
        return preferredElMin

    def isObservable(self, srcName,
                     minEl=20.,
                     noShadowingCheck=False):
        import numpy as np
        src = self.sourceDict[srcName]["fs"]
        # Check elevation and shadowing
        az, el = Observation.ObsCalSources.ObsCalSources.sourceAzEl(src)
        az = np.degrees(az)
        el = np.degrees(el)
        fShadow = self._shadowingChecker.isBlocked(az, el,
                                                   returnFraction=True)
        isObs = True
        # TODO: Elevation should be made also by looking ahead by
        #       several minutes. As a temporal workaround, just add
        #       2.5 degree to minEl
        if el < (minEl + 2.5) or el > 87:
            isObs = False
        if fShadow > self.maxShadowFraction:
            isObs = False
            values = (src.sourceName, fShadow * 100)
            msg = "Will skip source '%s' as %.1f percent of antennas will suffer from shadowing" % values
            self.logInfo(msg)

        values = (srcName, az, el, minEl, fShadow, isObs)
        msg = "[%s] (az,el)=(%6.1f,%5.1f) minEl=%.1f fShadow=%.1f isObservable=%s" % values
        self.logInfo(msg)

        if not isObs:
            return False
        return True

    def doCalObservations(self):
        self.doCalSource(self.bandpassCalSource, "bandpass")
        for ampCalSource in self.ampCalSources:
            self.doCalSource(ampCalSource, "flux")
        for secondaryAmpCalSource in self.secondaryAmpCalSources:
            self.doCalSource(secondaryAmpCalSource, "check")

        minEl = self._obsmode.getElevationLimit()
        minEl = math.degrees(CCL.SIConverter.toRadians(minEl))

        if self.sources != "":
            # DelayCal like execution
            srcNames = self.sources.split(",")
            for srcName in srcNames:
                if not self.isObservable(srcName, minEl=minEl,
                                         noShadowingCheck=True):
                    continue
                src = self.sourceDict[srcName]["fs"]
                self.doCalSource(src, "phase")
                self.writeLogEntry(srcName)
            return

        # Survey execution
        preferredElMin = self.determinePreferredMinimumElevation()
        nPhaseScansMade = 0
        while 1:
            srcNames = self.selectNextSources(N=3, minEl=minEl,
                                              preferredElMin=preferredElMin)
            if len(srcNames) == 0:
                break

            nObserved = 0
            for srcName in srcNames:
                if not self.isObservable(srcName, minEl=minEl):
                    continue
                src = self.sourceDict[srcName]["fs"]
                self.doCalSource(src, "phase")
                self.writeLogEntry(srcName)
                self.sourceDict.pop(srcName)
                nPhaseScansMade += len(self.bandList)
                nObserved += 1

            if nObserved == 0:
                # Guard for introducing an infinite loop.
                msg = "Although %d sources picked, none of them are observable" % (len(srcName))
                self.logInfo(msg)
                break

            if nPhaseScansMade >= 40:
                break

    def configureArray(self):
        """
        Call this after parsing command line options, to set manual mode array.
        Note this runs before the logger is ready.
        """
        arrayName = self.args.array
        # Override to add a support for OSS simulation
        if arrayName in ["ArrayOSS"]:
            from ObservingModeSimulation.ArraySimulator import SimulatedArray
            array_ = SimulatedArray()
            array_._arrayName = arrayName
            CCL.Global._array = array_
            CCL.Global.getArray = lambda : array_
            from Observation.AntennaPositionData import setupSimulatedTMCDBData
            setupSimulatedTMCDBData("7m", array=array_)
            array_.setCorrelatorType("ACA")
            ObsCalSourceCheck.arrayAlreadyDefined = lambda x: False
            ObsCalSourceCheck.configureArray = lambda x: (sys.stdout.write("%s\n" % x.args.array))
            return
        Observation.ObsCalBase.ObsCalBase.configureArray(self)

# Observation.ObsCalBase.checkArray()

survey = ObsCalSourceCheck()
survey.parseOptions()
survey.checkAntennas()
survey.startPrepareForExecution()
try:
    survey.generateTunings()
    survey.findPointFocusSource()
    survey.pickupCalSources()
except BaseException, ex:
    survey.logError(traceback.format_exc())
    survey.logException("Error in methods run during execution/obsmode startup", ex)
    survey.completePrepareForExecution()
    survey.closeExecution(ex)
    raise ex
survey.completePrepareForExecution()
survey.logInfo("Executing first pointing...")
survey.doPointing()
survey.logInfo("Executing second pointing -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing focus...")
survey.doFocus()
survey.logInfo("Executing third pointing after focus -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing Calibration observations...")
survey.doCalObservations()
survey.closeExecution()
