#*******************************************************************************
# ALMA - Atacama Large Millimeter Array
# Copyright (c) ESO - European Southern Observatory, 2011
# (in the framework of the ALMA collaboration).
# All rights reserved.
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#*******************************************************************************
# Optical Pointing Standard Script
# $Id: OffsetOpticalPointingScript.py 210969 2014-11-09 03:20:57Z nphillip $
#
# This script is an Observatory mode standard script.  It is used
# by AIV / CSV to asses the performance of the antennas
#

from CCL.Global import *
from math import degrees, radians
import CCL.utils
import Control
import ModeControllerExceptions
import ControlExceptions
import time
import os
import datetime
import Acspy.Common.Log, ACSLog
import Acspy.Common.ErrorTrace

logger = Acspy.Common.Log.Logger('OffsetOpticalPointingScript')
def logToOperator(msg, priority=ACSLog.ACS_LOG_INFO):
    import log_audience
    global logger
    logger.logNotSoTypeSafe(priority, msg, log_audience.OPERATOR)

logToOperator('Beginning the execution of the offset optical pointing script.')

# Get the Array and the Scheduling Block.
array = getArray()
array.beginExecution()

# Check that we have a single antenna in the array
antennaNames = array.antennas()
logToOperator("This array contains antennas " + str(antennaNames))
if len(antennaNames) != 1:
    errorMsg = "Array contains more than one antenna."
    logToOperator(errorMsg, ACSLog.ACS_LOG_CRITICAL)
    endExecution(Control.FAIL, errorMsg)

sb = getSB()

# This section of code creates a list containing all the targets in
# each group.
groupList = []
for group in sb.getGroups():
    referencesInThisGroup = CCL.utils.aggregate(group.OrderedTarget)
    targetsInThisGroup = sb.getReferencedTargets(referencesInThisGroup)
    groupList.append(targetsInThisGroup)

# Log some information from the scheduling block
logToOperator("This scheduling block contains " + \
               str(len(CCL.utils.aggregate(sb.FieldSource))) + \
               " stars")

# Optical Pointing observing parameters that come from the SB
optCameraSpec = sb.OpticalCameraSpec[0]
optPointingParams = sb.OpticalPointingParameters[0]
MinElevation = optPointingParams.elevationLimit.get()
# maximum source tracking error
TrackingTol = optPointingParams.antennaPositionTolerance.get()

# These are optional parameters, below are the defaults but they
# can be overridden with SB level paramters in the OT.

# Hardcoded behavior of the script
SNR_THRESHOLD = 10.0          # signal to noise ratio parameter
TRACK_TIMEOUT = 120           # maximum allowed time, in seconds, for
                              # the antenna to go "on-source".
                              
INITIAL_REPEAT = 65            # Observe the first source this many times
REPEAT_COUNT = 20             # Now observe all sources this many times
WRITE_FITS = 0

# Now get the expert paramters we need
for expertParameter in sb.getExpertParameters():
    if str(expertParameter.Keyword.lower()) == "initial repeat":
        INITIAL_REPEAT = int(expertParameter.Value)
    if str(expertParameter.Keyword.lower()) == "repeat count":
        REPEAT_COUNT = int(expertParameter.Value)
    if str(expertParameter.Keyword.lower()) == "write fits":
        WRITE_FITS = int(expertParameter.Value)

logToOperator('Initial Repeat Value set to: ' + str(INITIAL_REPEAT))
logToOperator('Repeat Count set to: ' + str(REPEAT_COUNT))
if WRITE_FITS is 0:
    logToOperator('FITS files will be not written')
elif WRITE_FITS is 1:
    logToOperator('FITS files will be written')
else:
    logToOperator('FITS files will be written every ' + str(WRITE_FITS) + ' scans')


# get the optical pointing observing mode object. The control
# subsystem must be operational for this to work. More specifically the
# relevant antenna, mount controller and optical telescope must be operational.
op = array.getOpticalPointingObservingMode()
opmc = op.getOpticalPointingModeController()
opt = opmc.getOpticalTelescope();
mc = opmc.getMountController()

# initialize the hardware (open the shutter etc.).
op.initializeHardware()

# Put the IR filter in for daytime observing
if str(optCameraSpec.filter) == 'day':
    logToOperator('Configuring optical telescope for day time observations.')
    opt.dayTimeObserving()
else:
    logToOperator('Optical telescope configured for night time observations.')
    opt.nightTimeObserving()

# set the timeout and tolerance for antenna motion.
mc.setTimeout(TRACK_TIMEOUT)
mc.setTolerance(TrackingTol)

# Create a directory for the TPOINT file and FITS images
mytime=datetime.datetime.now()
dir='Offset-Optical-Pointing-' + mytime.strftime('%Y-%m-%dT%H:%M:%S')
os.mkdir(dir)
os.chdir(dir);
filename='tpoint_' + mytime.strftime('%Y-%m-%d_%H%M%S') + '.dat'
f = file(filename, "w")
print >> f, filename
print >> f, ":NODA"
print >> f, ":ALLSKY"
print >> f, ":ALTAZ"
print >> f, "-23 04 23"
print >> f, " "

scan = 1
imagenum = 1

if optPointingParams.randomizeOrder:
    logToOperator('Randomization is not desirable for the Offset Optical Pointing mode.  Disregarding users request.', ACSLog.ACS_LOG_WARNING)

for group in groupList:
    initialSourceList = []
    for target in group:
         initialSourceList.append(sb.getReferencedFieldSource(target))

    sourceList = ([initialSourceList[0]]*(INITIAL_REPEAT - 1)) + \
                 (initialSourceList * REPEAT_COUNT)
    

    # We decide if we should observe a source based on if the first source
    # meets the criterea
    ra  = sourceList[0].sourceCoordinates.longitude.get()
    dec = sourceList[0].sourceCoordinates.latitude.get()
    (az, el) = mc.toAzElEquatorial(ra, dec)
    if el < MinElevation or el > radians(88.8):
        # Initial source out of range skip group
        logToOperator("Skipping this group as the elevation of the initial " +
                      "star " +
                      '%.2f'%degrees(el) +
                      " degrees is below the limit of " +
                      '%.2f'%degrees(MinElevation) + " degrees.",
                      ACSLog.ACS_LOG_WARNING)
        continue

    for source in sourceList:
        ra  = source.sourceCoordinates.longitude.get()
        dec = source.sourceCoordinates.latitude.get()
        logToOperator("Scan #" + str(scan) + " is '" + \
                      str(source.sourceName) + "' at " + \
                      mc.raDecToString(ra, dec))
        try:
            opmc.setDirection(ra, dec, \
                              source.pMRA, source.pMDec, source.parallax)
            time.sleep(0.)
            data = opmc.doExposure(optCameraSpec.minIntegrationTime);
            outfl="%.0f"%((data.endTime + data.startTime)/2.0)
            print >> f, '!', outfl, data.SNR, data.peakPositionX, data.peakPositionY
            print >> f, '!', degrees(data.pointingModelOffsetAz), degrees(data.pointingModelOffsetEl), \
                  degrees(data.CCDOffsetAz), degrees(data.CCDOffsetEl)
            print >> f,  degrees(data.sourceAZ.value), degrees(data.sourceEL.value), \
                  degrees(data.sourceAZ.value-data.offsetAZ.value), \
                  degrees(data.sourceEL.value-data.offsetEL.value)
            f.flush()                              
            op.beginScan(source);
            op.sendSubscanData(data)
            op.endScan();
            if (WRITE_FITS is not 0) and (scan%WRITE_FITS is 0):
                imageName = os.getcwd() + '/image-'+ '%03d'%imagenum +'.fits'
                logToOperator('Saving the image in ' + imageName);
                opmc.saveLastExposure(imageName);
                imagenum += 1
        except (ModeControllerExceptions.BadDataEx), e:
            logToOperator("Skipping this star as its parameters could not be determined", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        except (ModeControllerExceptions.TimeoutEx,
                ModeControllerExceptions.MountFaultEx), e:
            logToOperator("Skipping this star as the antenna could not point at it.", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        except (ModeControllerExceptions.HardwareFaultEx), e:
            logToOperator("Skipping this star as the was a problem with the hardware. Lets hope its intermittant.", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        except (ControlExceptions.HardwareErrorEx), e:
            logToOperator("Skipping this star as the was a problem with the hardware. Lets hope its intermittant.", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        scan += 1;
    print >> f, "! -------------- End of Group ----------------"
f.close();
logToOperator('Shutting down the optical telescope and antenna')
op.shutdownHardware()
array.endExecution(Control.SUCCESS, 'optical pointing')
