#*******************************************************************************
# ALMA - Atacama Large Millimeter Array
# Copyright (c) ESO - European Southern Observatory, 2011
# (in the framework of the ALMA collaboration).
# All rights reserved.
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#*******************************************************************************
# Optical Pointing Standard Script
# $Id: OpticalPointingScript.py 210969 2014-11-09 03:20:57Z nphillip $
#

from CCL.Global import *
from math import degrees, radians
import CCL.utils
import Control
import ModeControllerExceptions
import ControlExceptions
import Acspy.Common.ErrorTrace
import time
import random
import os
import datetime
import Acspy.Common.Log, ACSLog

logger = Acspy.Common.Log.Logger('OpticalPointingScript')
def logToOperator(msg, priority=ACSLog.ACS_LOG_INFO):
    import log_audience
    global logger
    logger.logNotSoTypeSafe(priority, msg, log_audience.OPERATOR)

logToOperator('Beginning the execution of the optical pointing script.')

# Get the Array and the Scheduling Block.
array = getArray()
array.beginExecution()

# Check that we have a single antenna in the array
antennaNames = array.antennas()
logToOperator("This array contains antennas " + str(antennaNames))
if len(antennaNames) != 1:
    errorMsg = "Array contains more than one antenna."
    logToOperator(errorMsg, ACSLog.ACS_LOG_CRITICAL)
    endExecution(Control.FAIL, errorMsg)

sb = getSB()

# Log some information from the scheduling block
logToOperator("This scheduling block contains " + \
              str(len(CCL.utils.aggregate(sb.FieldSource))) + \
              " stars")

# Optical Pointing observing parameters that come from the SB
optCameraSpec = sb.OpticalCameraSpec[0]
optPointingParams = sb.OpticalPointingParameters[0]
MinElevation = optPointingParams.elevationLimit.get()
# maximum source tracking error
TrackingTol = optPointingParams.antennaPositionTolerance.get()

# Hardcoded behavior of the script
SNR_THRESHOLD = 10.0          # signal to noise ratio parameter
TRACK_TIMEOUT = 120           # maximum allowed time, in seconds, for
                              # the antenna to go "on-source".

# get the optical pointing observing mode object. The control
# subsystem must be operational for this to work. More specifically the
# relevant antenna, mount controller and optical telescope must be operational.
op = array.getOpticalPointingObservingMode()
opmc = op.getOpticalPointingModeController()
opt =  opmc.getOpticalTelescope();
mc = opmc.getMountController()

# initialize the hardware (open the shutter etc.).
op.initializeHardware()

# Put the IR filter in for daytime observing
if str(optCameraSpec.filter) == 'day':
    logToOperator('Configuring optical telescope for day time observations.')
    opt.dayTimeObserving()
else:
    logToOperator('Optical telescope configured for night time observations.')
    opt.nightTimeObserving()

# set the timeout and tolerance for antenna motion.
mc.setTimeout(TRACK_TIMEOUT)
mc.setTolerance(TrackingTol)

# Create a directory for the TPOINT file and FITS images 
mytime=datetime.datetime.now()
dir='Optical-Pointing-' + mytime.strftime('%Y-%m-%dT%H:%M:%S')
os.mkdir(dir)
os.chdir(dir);
filename='tpoint_' + mytime.strftime('%Y-%m-%d_%H%M%S') + '.dat'
f = file(filename,"w")
print >> f, filename
print >> f, ":NODA"
print >> f, ":ALLSKY"
print >> f, ":ALTAZ"
print >> f, "-23 04 23"
print >> f, " "

#Advanced Parameters
WRITE_FITS = 0
for expertParameter in sb.getExpertParameters():
    if str(sb.getExpertParameters()[0].Keyword.lower()) == "write fits":
        WRITE_FITS =  int(sb.getExpertParameters()[0].Value)

if WRITE_FITS is 0:
    logToOperator('FITS files will be not written')
elif WRITE_FITS is 1:
    logToOperator('FITS files will be written')
else:
    logToOperator('FITS files will be written every ' + str(WRITE_FITS) + ' scans')

scan = 1
imagenum = 1
#
# Randomization 
#
sourceList = sb.getTargets()
if optPointingParams.randomizeOrder:
    logToOperator('Randomizing pointing targets.')
    random.shuffle(sourceList)
	
for target in sourceList:
    # get the source
    source = sb.getReferencedFieldSource(target)
    ra = source.sourceCoordinates.longitude.get()
    dec = source.sourceCoordinates.latitude.get()
    logToOperator("Scan #" + str(scan) + " is '" + \
                  str(source.sourceName) + "' at " + \
                  mc.raDecToString(ra, dec))

    (az, el) = mc.toAzElEquatorial(ra, dec)
    if el > MinElevation and el < radians(88.8):
        logToOperator("Its approximate (Az, El) is " + mc.azElToString(az, el))
        try:
            opmc.setDirection(ra, dec, \
                              source.pMRA,source.pMDec, source.parallax)
            time.sleep(3.)
            data = opmc.doExposure(optCameraSpec.minIntegrationTime);
            outfl="%.0f"%((data.endTime + data.startTime)/2.0)
            print >> f, '!', outfl, data.SNR, data.peakPositionX, data.peakPositionY
            print >> f, '!', degrees(data.pointingModelOffsetAz), degrees(data.pointingModelOffsetEl), \
                  degrees(data.CCDOffsetAz), degrees(data.CCDOffsetEl)
            print >> f,  degrees(data.sourceAZ.value), degrees(data.sourceEL.value), \
                  degrees(data.sourceAZ.value-data.offsetAZ.value), \
                  degrees(data.sourceEL.value-data.offsetEL.value)
            f.flush()
            op.beginScan(source);
            op.sendSubscanData(data)
            op.endScan();
            if (WRITE_FITS is not 0) and (scan%WRITE_FITS is 0):
                imageName = os.getcwd() + '/image-'+ "%03d"%imagenum +'.fits'
                logToOperator('Saving the image in ' + imageName);
                opmc.saveLastExposure(imageName);
                imagenum += 1
        except (ModeControllerExceptions.BadDataEx), e:
            logToOperator("Skipping this star as its parameters could not be determined", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        except (ModeControllerExceptions.TimeoutEx,
                ModeControllerExceptions.MountFaultEx), e:
            logToOperator("Skipping this star as the antenna could not point at it", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
        except (ControlExceptions.HardwareErrorEx), e:
            logToOperator("Skipping this star as the was a problem with the hardware. Lets hope its intermittant.", ACSLog.ACS_LOG_WARNING)
            logger.logInfo("Underlying error message was: ")
            eth = Acspy.Common.ErrorTrace.ErrorTraceHelper(e.errorTrace);
            logger.logInfo(eth.errorTraceToString(e.errorTrace, ""))
    else:
        logToOperator("Skipping this star as its elevation of " + '%.2f'%degrees(el) + " degrees is below the limit of " + '%.2f'%degrees(MinElevation) + " degrees.", ACSLog.ACS_LOG_WARNING)
    scan += 1;
f.close();
logToOperator('Shutting down the optical telescope and antenna')
op.shutdownHardware()
array.endExecution(Control.SUCCESS, 'optical pointing')

