#! /usr/bin/env python
# ALMA - Atacama Large Millimeter Array
# (c) Associated Universities Inc., 2009 - 2011
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

#************************************************************************
#   NAME PhaseDifferentialIntererometry.py
# 
#   UPDATE THIS! SYNOPSIS This script is intended to meet the needs of multi-target, survey
#       style observations which have a common spectral configuration for all
#       science targets and a shared primary phase calibrator.  
# 
#------------------------------------------------------------------------
#

import Control
import Observation.SchedulingBlock
from Observation.PhaseDifferentialSBExecutionMode import PhaseDifferentialSBExecutionMode
from Observation.SBExecutionMode import SBExecutionError
from Observation.Global import simulatedArray, addClassToMode, OperatorLogger
from CCL.Global import *
from CCL.APDMSchedBlock import SpectralSpec
from CCL.ObservingModeBase import SBOutOfTime
from PyDataModelEnumeration import PyCalibrationDevice
from Observation.SBRatioCalTarget import SBRatioCalTarget
from Observation.PointingCalTarget import PointingCalTarget
from Observation.TargetUtility import *
from Observation.SBExecState import SBExecState
import math
from time import time

logger = OperatorLogger()

sb = Observation.SchedulingBlock.SchedulingBlock()
# be ready for the worst
status = (Control.FAIL, "SB failed")

B_max = 450.0 # length, in meters, of the longest baseline in the array to be executed.  Could either
              # get this from the resolution requirement or from the scheduler with the configuration

# Get expert parameters as needed here
#
# Get expert parameters as needed here
#
numPhaseCal     = int(sb.getExpertParameter('NumPhaseCal',  default=1))
logger.logInfo('NumPhaseCal set to  %s' % numPhaseCal)
numAmplitudeCal = int(sb.getExpertParameter('NumAmplitudeCal', default=1))
logger.logInfo('NumAmplitudeCal set to  %s' % numAmplitudeCal)
numBandpassCal  = int(sb.getExpertParameter('NumBandpassCal', default=1))
logger.logInfo('NumBandpassCal set to  %s' % numBandpassCal)
numPointingCal  = int(sb.getExpertParameter('NumPointingCal', default=1))
numFocusCal     = int(sb.getExpertParameter('NumFocusCal', default=1))
elLimit         = float(sb.getExpertParameter('ElLimit', default=20.))
maxPointingSeparation = float(sb.getExpertParameter('MaxPointingSeparation', default=25.))
sourceCycle     = float(sb.getExpertParameter('SourceCycleTime', default=0))
if sourceCycle <= 0:
    sourceCycle = 12.0/(B_max*7.29e-5)
logger.logInfo('Source Cycle Time: %f s' % sourceCycle)

phaseDifferentialCalCycleTime = float(sb.getExpertParameter('PhaseDiffCalCycleTime', default=1500.))

logger.logInfo('PhaseDiffCalCycleTime set to  %s' % phaseDifferentialCalCycleTime ) 
phaseDifferentialNumCycles    = int(sb.getExpertParameter('NumPhaseDiffCalCycles',   default=3))
logger.logInfo('NumPhaseDiffCalCycles set to  %s' % phaseDifferentialNumCycles ) 
sessionControl  = int(sb.getExpertParameter('SessionControl', default=0))

# Get reference to the array and start the exec block
array = getArray()
if sessionControl > 0:

    sbe = SBExecState(array, sb)
    state_tuple  = sbe.getStateFromFile()
    if state_tuple is None:
        print "First SB Execution"
        numBandpassDone = 0
        numAmplitudeDone = 0
    else:
        (recoveredObjects,) = state_tuple 
        print 'SB alreadyExecuted, recovered objects:'
        print recoveredObjects
        numBandpassDone = recoveredObjects['numBandpassDone']
        numAmplitudeDone = recoveredObjects['numAmplitudeDone']
else:
    numBandpassDone = 0
    numAmplitudeDone = 0

array.beginExecution()

# Get the Observing Mode and extend it
obsmode = array.getInterferometryObservingMode()
addClassToMode(obsmode, PhaseDifferentialSBExecutionMode)

obsmode.setPhaseDifferentialCalCycles(cycleTime= phaseDifferentialCalCycleTime,
                                      numCycles = phaseDifferentialNumCycles)
# Just start on the sky, to be sure. Should be in the mode defaults.
obsmode.setCalibrationDevice(PyCalibrationDevice.NONE)

# Set observing mode restrictions
obsmode.setElevationLimit('%f deg' % elLimit)
logger.logInfo("Setting elevation limit to %f degrees" % elLimit)
endTime = sb.getMaximumExecutionTime()
logger.logInfo("Setting maximum execution time to %s seconds or %s minutes" % (endTime,endTime/60.0))
obsmode.setMaximumSchedBlockTime(endTime)
logger.logInfo("Setting max. pointing separation  to %f degrees" % maxPointingSeparation)
PointingCalTarget.setMaximumSeparation('%f deg' % maxPointingSeparation )

# This should be the default for real interferometry: we use reference positions only for AtmCals.
for target in sb.getTargetList():
    target.setUseReferencePosition(target in sb.getAtmCalTargetList())
# Get group list
groupList = sb.getGroupList()
#
# Check the visibility of science targets in all groups, (they should need no query expansion)
for group in groupList:
    if not obsmode.checkScienceTargetVisibility(group):
        logger.logError('Science targets in group %d are not visible, try later or tomorrow.' \
                        %(groupList.index(group)+1))
        raise SBExecutionError, "No visible science target"
logger.logInfo("Science targets are up")
#
# Expand source queries, then associate atm and pointing targets with other targets:
for group in groupList:
    logger.logInfo("Preparing targets for Observing Group %d" % (groupList.index(group)+1))
    t0 = time()
    group.prepareTargets()
    logger.logInfo('Preparation of targets took %2.2f seconds' %(time()-t0))

#
# fill in Atm and SB Ratio Cal targets:
t0 = time()
sb.createAtmCalTargets()
sb.createSBRatioCalTargets()
logger.logInfo('Preparation of Atm and SBRatio Cal targets took %2.2f seconds' %(time()-t0))

t0 = time()
obsmode.adjustDoppler(sb.getTargetList(), sb.getScienceTargetList())
logger.logInfo('Doppler adjustment took %2.2f seconds' %(time()-t0))
    
#
# be ready for the worst
status = (Control.FAIL, "SB failed")
#
try:
    # adjust Doppler once using the science targets (only the first one).
    # obsmode.adjustDoppler(sb.getTargetList(), sb.getScienceTargetList())
    # The first group is for SB calibrators, we don't expect any science target.
    group = groupList[0]
    groupNumber = groupList.index(group)+1
    logger.logInfo("Start Observing Group %d (Calibrations)" % groupNumber)
    #
    # pointing may be executed in association with each target
    obsmode.executeCalTargetList(group.getFocusCalTargetList(), minimumNumber=numFocusCal)
    # note bandpasses are normally in group 2 for this mode.
    if numBandpassDone < numBandpassCal:
        numBandpassDone, numBandpassInCleanup = \
                         obsmode.executeCalTargetListSpec(group.getBandpassCalTargetList(),
                                                          minimumNumber=numBandpassCal,
                                                          deferToCleanupList=True)
        logger.logInfo("Bandpass Targets: %d executed, %d in Cleanup List" %\
                       (numBandpassDone, numBandpassInCleanup))

    obsmode.executeCalTargetList(group.getDelayCalTargetList())
    if numAmplitudeDone < numAmplitudeCal:
        numAmplitudeDone, numAmplitudeInCleanup = \
                          obsmode.executeCalTargetList(group.getAmplitudeCalTargetList(),
                                                       minimumNumber=numAmplitudeCal,
                                                       deferToCleanupList=True)
        logger.logInfo("Amplitude Cal Targets: %d executed, %d in Cleanup List" \
                       %(numAmplitudeDone, numAmplitudeInCleanup))
                   
    # Now the rest of the groups should contain science targets
    # We start with group 2, group 1 is a container for the phase differential targets.
    for group in groupList[2:]:
        groupNumber = groupList.index(group)+1
        # The primary phase calibrator is assumed to be the one with the
        # shortest cycle time in the group.  This effectivly sets the scan
        # duration.
        primaryPhaseCal = obsmode.getPrimaryPhaseCal(group)
        if primaryPhaseCal is None:
            logger.logError("No phase calibrator available in group %s" % groupNumber)
            raise SBExecutionError('No Phase Calibrator')
        
        logger.logInfo("Start Observing Group %d (Science)" % groupNumber)
        # the primaryPhaseDifferentialCalTarget is defined as the cal target in the phase diff group
        # which has the SAME spectralSpec as the primaryPhaseCal.
        primaryPhaseDifferentialCalTarget = obsmode.selectPrimaryPhaseDifferentialTarget(\
            groupList[1].getBandpassCalTargetList(),
            primaryPhaseCal)
        if primaryPhaseDifferentialCalTarget is None:
            logger.logError("No phase differential calibrator available in group 2 (Ph.Diff.)")
            raise SBExecutionError('No phase differential calibrator')
        
        # A group is defined to be complete when all science targets within
        # the group are complete (or unobservable)
        obsmode.assignSubcycleTime(group.getScienceTargetList(),sourceCycle)
        while not group.isComplete(obsmode):
            obsmode.executePhaseDifferentialCalGroupIfNeeded(\
                groupList[1].getBandpassCalTargetList(),\
                primaryPhaseDifferentialCalTarget)
            obsmode.executeCalTargetList(group.getFocusCalTargetList())
            obsmode.executeCalTargetList(group.getAmplitudeCalTargetList())
            obsmode.executeCalTargetList(group.getBandpassCalTargetList())
            obsmode.executeCalTargetList(group.getPhaseCalTargetList(),minimumNumber=numPhaseCal)
            obsmode.executeScienceTargetList(group,[primaryPhaseCal])
            primaryPhaseCal.execute(obsmode)
            if not primaryPhaseCal.isObservable(obsmode,
                                                duration=primaryPhaseCal.getCycleTime()):
                logger.logWarning("Attempting to find a new primary phase calibrator as %s will soon be setting"\
                                  % primaryPhaseCal.getSource().sourceName)
                primaryPhaseCal = obsmode.getPrimaryPhaseCal(group, primaryPhaseCal)
                logger.logWarning("The new primary phase calibrator is %s" \
                                  % primaryPhaseCal.getSource().sourceName)

        # Group is complete: remove the primaryPhaseCal from cleanup:
        logger.logInfo("Group number %s completed." % groupList.index(group))
        
        obsmode.removeTargetFromCleanupList(primaryPhaseCal)
                         
    # All groups are complete: execute the cleanup (e.g. bandpass or amplitude cals)
    logger.logInfo("All groups are completed.")
    logger.logInfo("Executing cleanup list.")
    obsmode.executeCleanupList()
    status = (Control.SUCCESS, "Successful Completion")
                    
except SBOutOfTime:
    #
    # Ok there should be just enough time for the cleanup list now
    logger.logWarning("Running short of time, forcing cleanup!")
    logger.logInfo("Executing cleanup list.")
    obsmode.executeCleanupList()
    status = (Control.SUCCESS, "Successful Completion")
    
except Exception, ex:
    logger.logError("Exception caught: %s" % ex)
    status = (Control.FAIL, "SB failed")
    raise
    
finally:
    try:
        obsmode.resetLimits()
        if sessionControl>0:
            logger.logInfo("Saving SB State.")
            sbe.saveStateToFile(({'numBandpassDone': numBandpassDone, 'numAmplitudeDone': numAmplitudeDone}))

    except:
        logger.logError("Could not reset limits")

    array.endExecution(*status)
           
#
# ___oOo___
