#! /usr/bin/env python
# ALMA - Atacama Large Millimeter Array
# (c) Associated Universities Inc., 2009 - 2011
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that i1t will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: StandardSingleDish.py 237411 2016-11-09 19:18:54Z ahirota $"

#************************************************************************
#   NAME StandardSingleDish.py
#
#   SYNOPSIS This script is inteded to provide basic single dish
#            observations capabilities of single pointing targets.
#
#------------------------------------------------------------------------

#
# forcing global imports is due to an OSS problem
#
# import Control
from CCL.Global import getArray
from CCL.ObservingModeBase import SBOutOfTime
import CCL.APDMSchedBlock
from PyDataModelEnumeration import PyCalibrationDevice
global Observation
import Observation.SchedulingBlock
from Observation.SBExecutionMode import SBExecutionMode, SBExecutionError
from Observation.Global import addClassToMode
from Observation.ScanList import ScanList
from Observation.SBExecState import SBExecState
from Observation.AtmCalTarget import AtmCalTarget
import numpy as np

# Ugly hack to get nice logging when we're not written as a class
class StandardSingleDish(Observation.SSRLogger.SSRLogger):
    def __init__(self):
        Observation.SSRLogger.SSRLogger.__init__(self, self.__class__.__name__)
logger = StandardSingleDish()

sb = Observation.SchedulingBlock.SchedulingBlock()

# Get expert parameters as needed here
useScanSequence = bool(int(sb.getExpertParameter('useScanSequence', default=False))) # currently screwed-up with dual mode
sessionControl  = int(sb.getExpertParameter('SessionControl', default=0))
elLimit = float(sb.getExpertParameter('ElLimit', default=20.))
useOrthogonalScans = bool(int(sb.getExpertParameter('useOrthogonalScans', default="0")))
logger.logInfo('Session control %d' % sessionControl)
logger.logInfo("useScanSequence=%s" % useScanSequence)
logger.logInfo("useOrthogonalScans=%s" % useOrthogonalScans)

# Get reference to the array and start the exec block
array = getArray()
array.beginExecution()

# None means OK. If an error occurs the status is replaced, in the catch block, by the exception.
status = None
obsmode = None
try:
    # Get the Observing Mode and set the maximum execution time
    obsmode = array.getInterferometryObservingMode()
    # Add the SB execution methods
    addClassToMode(obsmode, SBExecutionMode)
    logger.logInfo("ASDM UID = %s" % (obsmode.getExecBlockUID()))
    sbe = SBExecState(array, sb)
    if sessionControl:
        sbe.startSession()
    obsmode.addSBExecState(sbe)

    #Just start on the sky, to be sure.  Should be in the mode defaults
    obsmode.setCalibrationDevice(PyCalibrationDevice.NONE)

    # Set observing mode restrictions
    logger.logInfo("Setting elevation limit to %f degrees" % elLimit)
    obsmode.setElevationLimit('%f deg' % elLimit)
    endTime = sb.getMaximumExecutionTime()
    logger.logInfo("Setting maximum execution time to %s seconds or %s minutes" % (endTime, endTime / 60.0))
    obsmode.setMaximumSchedBlockTime(endTime)

    groupList = sb.getGroupList()
    if len(groupList) < 2:
        raise SBExecutionError("Single dish SB observation requires two observing groups")

    # Check the visibility of science targets in all groups, (they should need no query expansion)
    for group in groupList:
        if not obsmode.checkScienceTargetVisibility(group):
            logger.logError('Science targets in group %d are not visible, try later or tomorrow.' % (group.groupIndex))
            raise SBExecutionError("No visible science target")

    for group in groupList:
        # Perform doppler correction for spectral specs associated with this group (ICT-5506).
        # The doppler correction is made by overriding the BB center frequencies.
        sb.adjustDopplerByGroup(obsmode, group)

        # Enable sideband separation for DSB bands by generating a set of targets with
        # slightly shifted LO1 frequencies. This has to be made after the doppler correction.
        sb.populateLOShiftedTarget(obsmode, group)

        # Then, resolve queries.
        group.prepareTargets(assocPntToSciTarget=True)
        sb.updateTargetList()
        # And, create associated ATM targets (all spectral specs should be in TOPO frame)
        group.createAtmCalTargets(isSingleDish=True)
        # SBR targets are omitted from normal science SB executions, but
        # if it is requested to revert, here is the place to add SBR targets.
        ### group.createSBRatioCalTargets()
        sb.updateTargetList()

    sb.setTargetsId()

    # Diasable online procesing of SBR, as TelCal's defaults are generally more reliable for a small array of antennas.
    logger.logInfo('Disabling online sideband ratio processing')
    for target in sb.getSBRatioCalTargetList():
        target.setOnlineProcessing(False)

    if sessionControl:
        # Recycle Field Sources
        obsmode.sbe.recycleFieldSources()
        # # Recycle attenuator settings
        # # Mark _ifSkyAttenSet to re-use previous attenuator optimizations.
        # # This relies on the fact that attenuator setting name is identical as spectral spec name.
        # obsmode.sbe.recycleAttenSettings()
        # Recycle science target integration time and offset index
        obsmode.sbe.restoreScienceTargetParameters(restoreIntegrationTime=True)

    # This should be the default for real interferometry: we use reference positions only for AtmCals and science targets.
    for target in sb.getTargetList():
        target.setUseReferencePosition(target in sb.getAtmCalTargetList() + sb.getScienceTargetList())

    # (CSV-2987) A switch for disabling the use of reference position for SD obs.
    for iT, target in enumerate(sb.getScienceTargetList()):
        useRef = bool(int(target.getObservingParameter("UseReference", default=1)))
        logger.logInfo("[%2d] [%s] UseReference=%s" % (iT, target, useRef))
        if not useRef:
            logger.logInfo("Disabling the use of reference position for '%s' ..." % target)
            target.setUseReferencePosition(False)

    # SCIREQ-800: Enable orthogonal scans
    for target in sb.getScienceTargetList():
        target.enableOrthogonalScans(useOrthogonalScans)

    group = groupList[0]
    logger.logInfo("Start Observing Group %d (Calibrations)" % group.groupIndex)

    # Use dual mode always -- this is single dish! 16ms default sampling
    for target in sb.getTargetList():
        ss = target.getSpectralSpec()
        try:
            sqlTest = ss.SquareLawSetup.integrationDuration
            logger.logInfo("SpectralSpec '%s' already has a SquareLawSetup" % str(ss.name))
        except AttributeError:
            ss.SquareLawSetup = CCL.APDMSchedBlock.SquareLawSetup()
            ss.SquareLawSetup.integrationDuration.set(0.016)
            logger.logInfo("SpectralSpec '%s' had no SquareLawSetup, so added one" % str(ss.name))

    nFocus, nFocusCleanup = obsmode.executeCalTargetList(group.getFocusCalTargetList()) #, minimumNumber=numFocusCal)
    logger.logInfo("Focus Targets: %d executed, %d in Cleanup List" % (nFocus, nFocusCleanup))

    # Now the rest of the groups should contain science targets
    # This won't do scan sequences, testing first
    for group in groupList[1:]:
        groupNumber = group.groupIndex
        logger.logInfo("[Group %d] BEGIN" % (groupNumber))
        while not group.isComplete(obsmode):
            obsmode.executeCalTargetList(group.getDelayCalTargetList(), scanList=None)

            scienceTargetList = group.getScienceTargetList()
            scienceTargetList = [t for t in scienceTargetList if not t.observationComplete()]
            nRemain = len(scienceTargetList)

            # Exclude targets not observable right now
            scienceTargetList = [t for t in scienceTargetList if t.isObservable(obsmode)]
            if len(scienceTargetList) == 0:
                raise SBExecutionError("All of the remaining science targets in group%d are not observable: aborting execution" % (groupNumber))

            # Pick up a target that has achived minimum integration so far.
            intTimeList = [t.getCurrentIntegrationTime() for t in scienceTargetList]
            scienceTarget = scienceTargetList[np.argmin(intTimeList)]

            t0 = obsmode.getElapsedTime()
            logger.logInfo("[Group %d] [%s] Starting at %.1f" % (groupNumber, scienceTarget, t0))
            cleanupList = []
            while 1:
                # Execute associated cal targets (pointing and ATM), if needed.
                assocCalTargets = scienceTarget.getAssociatedCalTarget()
                [target.executeIfNeeded(obsmode) for target in assocCalTargets]

                # Define a list of ATM targets, as ATM cycle time is the only thing that
                # limits the length of science target execution.
                atmcalList = [assocTarget for assocTarget in assocCalTargets if isinstance(assocTarget, AtmCalTarget)]
                maximumTime = 450.
                if len(atmcalList) > 0:
                    timeUntilNextATM = min([atm.getNextRequiredTime() for atm in atmcalList]) - obsmode.getElapsedTime()
                    logger.logInfo("timeUntilNextATM = %8.3f" % (timeUntilNextATM))
                    maximumTime = min(maximumTime, timeUntilNextATM)
                logger.logInfo("maximumTime = %8.3f" % (maximumTime))

                if scienceTarget.isMapStart():
                    # ICT-8004: for raster scan with two OFFs, have to configure
                    # reference index at the every beginning of map scan.
                    sb.configureOffPointIndex(scienceTarget)

                # Disable SBOutOfTime, so that raster map will not be interrupted.
                obsmode.setMaximumSchedBlockTime(None)
                scienceTarget.execute(obsmode, maximumTime=maximumTime)
                if scienceTarget.isMapStart():
                    scienceTarget.alternateScanningDirection(maxAspectRatio=2.)

                # ICT-6490: Make sure that ATM scans bracket science scans.
                for atmTarget in atmcalList:
                    atmTarget.execute(obsmode)

                # Enable SBOutOfTime again
                obsmode.setMaximumSchedBlockTime(endTime)

                t1 = obsmode.getElapsedTime()
                logger.logInfo("[Group %d] [%s] Elapsed %.1f [sec]" % (groupNumber, scienceTarget, t1 - t0))
                if scienceTarget.observationComplete():
                    logger.logInfo("[Group %d] [%s] Obsevation complete for this target." % (groupNumber, scienceTarget))
                    break
                tRemain = obsmode.getTimeBeforeCleanup()
                if tRemain and tRemain < 0:
                    logger.logInfo("[Group %d] Running out of time for SB execution." % (groupNumber))
                    break
                if nRemain > 1 and (t1 - t0) > 1200:
                    logger.logInfo("[Group %d] [%s] Will switch to another source." % (groupNumber, scienceTarget))
                    break

            obsmode.checkElapsedTime()

        logger.logInfo("[Group %d] END" % (groupNumber))
except SBOutOfTime:
    # TODO: not doing clean up, at this moment. (but, it is not required?)
    logger.logWarning("Running short of time, forcing cleanup!")
except Exception, ex:
    logger.logError("Exception caught: %s" % ex)
    status = ex
    raise
finally:
    if "group" in locals():
        for target in group.getScienceTargetList():
            cIntegTime = target.getCurrentIntegrationTime()
            msg = "Achieved integration time=%5d secs " % (cIntegTime)
            msg += "(target value=%5d secs)" % (target.getIntegrationTime())
            logger.logInfo("[%s] %s" % (target, msg))
    if sessionControl:
        sbe.endSBExecution()
    array.endExecution(status)
    try:
        if obsmode:
            obsmode.cleanupObsMode(array, logger)
    except:
        import traceback
        logger.logWarning("Failed to execute cleanupObsMode()")
        logger.logWarning(traceback.format_exc())
#
# ___oOo___
