#! /usr/bin/env python
global sys
import sys
global os
import os
global time
import time
global traceback
import traceback
global np
import numpy as np
global Control
import Control
global CCL
from CCL.Global import getArray
import CCL.APDMSchedBlock
global SBOutOfTime
from CCL.ObservingModeBase import SBOutOfTime
global Observation
import Observation.SchedulingBlock
global SBExecutionError
from Observation.SBExecutionMode import SBExecutionError
import Observation.SSRLogger
import Observation.SSRTuning
import Observation.PointingCalTarget
import Observation.ScienceTarget
import Observation.ObsTarget
global MD
import Observation.MixerDetuning as MD
global CalibratorCatalog
from Observation.CalibratorCatalog import CalibratorCatalog
global TelCalParameters
import TelCalParameters


global performance
def performance(f):
    import sys
    def d(*args, **kwargs):
        t0 = time.time()
        r = f(*args, **kwargs)
        if len(args) == 0:
            return r
        self = args[0]
        if hasattr(self, "logInfo"):
            self.logInfo('[performance] [%s] %8.3f [sec]' % (f.__name__, time.time() - t0))
        return r
    return d


global getBand
def getBand(spectralSpec):
    bandName = str(spectralSpec.FrequencySetup.receiverBand)
    bandNum = int(bandName[-2:])
    return bandNum


global getMixerMode
def getMixerMode(target):
    mixerMode = target.getObservingParameter("mixerMode", default="normal")
    if mixerMode.lower() not in ["normal", "solar1", "solar2"]:
        raise Exception("Unsupported mixerMode (%s) specified to '%s'" % (mixerMode, target))
    return mixerMode


global getAttenuatorShifts
def getAttenuatorShifts(target):
    ifswAttenShift = float(target.getObservingParameter("ifswAttenShift", default=0.))
    ifprAttenShift = float(target.getObservingParameter("ifprAttenShift", default=0.))
    return ifswAttenShift, ifprAttenShift


class StandardSolarObs(Observation.SSRLogger.SSRLogger):
    """
    Limitations:
        - Not supporting scan sequence.
    """
    def __init__(self, array):
        Observation.SSRLogger.SSRLogger.__init__(self, self.__class__.__name__)
        self._array = array
        self._mode = None
        self._detune = None
        self._detuneAntennaList = array.antennas()
        self._cc = CalibratorCatalog("observing", useSSRLogger=True)
        self._cc.minSeparationDict["Sun"] = 2.
        self._cc.minSeparationDict["Moon"] = 2.

        self._sb = None
        self._tm_str = time.strftime('%Y%m%d_%H%M%S', time.gmtime())
        self._referenceAttenuatorSetting = None

        self._logPower = False
        self._logSISBias = False

        self._noQuery = os.getenv("OSS_NO_QUERY", None) not in [None, "", "0"]
        self._noPointingQuery = os.getenv("OSS_NO_PNT_QUERY", None) not in [None, "", "0"]
        self._noPointing = False

    def loadSB(self, sbFileName=None):
        if sbFileName:
            sb = Observation.SchedulingBlock.SchedulingBlock(sbFileName)
        else:
            sb = Observation.SchedulingBlock.SchedulingBlock()
        self._sb = sb
        self.logInfo('Running %s scheduling block' % sb._controlSB.name)

        # Read expert parameters
        self._elLimit = float(sb.getExpertParameter('ElLimit', default=10.))
        self._maxPointingSeparation = float(sb.getExpertParameter('MaxPointingSeparation', default=25.))
        self._useDualMode = bool(int(sb.getExpertParameter('useDualMode', default=True)))
        self._sessionControl = int(sb.getExpertParameter('SessionControl', default=0))
        self._logPower = bool(int(sb.getExpertParameter("LogPower", default=0)))
        self._logSISBias = bool(int(sb.getExpertParameter("LogSISBias", default=0)))
        activeSun = bool(int(sb.getExpertParameter("activeSun", default=0.)))
        if activeSun:
            solarMode = "solar2"
        else:
            solarMode = "normal"
        solarMode_ = sb.getExpertParameter("SolarMode", default="").strip()
        if solarMode_ != "":
            solarMode = solarMode_
        self._solarMode = solarMode

        self.logInfo("Session Control = %s" % (self._sessionControl))
        self.logInfo("activeSun = %s" % (activeSun))
        self.logInfo("solarMode = %s" % (solarMode))
        self.logInfo("_logPower = %s" % (self._logPower))
        self.logInfo("_logSISBias = %s" % (self._logSISBias))

        ifLevel = sb.getExpertParameter('ifLevel', default="")
        self._ifLevel = None if ifLevel == "" else float(ifLevel)
        self.logInfo("_ifLevel = %s" % (self._ifLevel))

        bbLevel = sb.getExpertParameter('bbLevel', default="")
        self._bbLevel = None if bbLevel == "" else float(bbLevel)
        self.logInfo("_bbLevel = %s" % (self._bbLevel))

        self._useControlMD = hasattr(Control, "MD1")
        if int(sb.getExpertParameter("UseControlMD", default=1)) == 0:
            self._useControlMD = False
        self.logInfo("_useControlMD = %s" % (self._useControlMD))

        return sb

    def getSB(self):
        if self._sb is None:
            raise SBExecutionError("Scheduling block not loaded")
        return self._sb

    def startExecution(self):
        self._array.beginExecution()
        self._setupObservingMode()

    def _setupObservingMode(self):
        from Observation.Global import addClassToMode
        from PyDataModelEnumeration import PyCalibrationDevice
        from Observation.FrontEndAttenuation import FrontEndAttenuation
        from Observation.SBExecState import SBExecState
        from Observation.SBExecutionMode import SBExecutionMode

        if self._mode:
            self.logInfo('ObservingMode has already configured.')
            return
        obsmode = self._array.getInterferometryObservingMode()
        addClassToMode(obsmode, SBExecutionMode)
        self.logInfo("ASDM UID = %s" % (obsmode.getExecBlockUID()))
        self._mode = obsmode
        sbe = SBExecState(self._array, self._sb)
        if self._sessionControl:
            sbe.startSession()
        self._mode.addSBExecState(sbe)

        arrayName = self._array._arrayName.replace("CONTROL/", "")
        self.logInfo("Setting TelCal parameters for array %s: spectrum=True" % (arrayName))
        if not MD.isSimulatedArray():
            tcParameters = TelCalParameters.TelCalParameters(arrayName)
            # spectrum is to enable channel-by-channel bandpass result. Also binningFactor can be specified if desired.
            tcParameters.setCalibParameter('spectrum', True)

        # Just start on the sky, to be sure. Should be in the mode defaults.
        self._mode.setCalibrationDevice(PyCalibrationDevice.NONE)

        self.logInfo("Setup FrontEndAttenuation...")
        self._fea = FrontEndAttenuation(self._mode)
        if self._mode.sbe.getSessionState() and not self._mode.sbe.isFirstTime():
            self.logInfo("Using sessions: keep previous attenuator settings...")
        else:
            self.logInfo("Not 2nd SB in a session: clear previous attenuator settings...")
            if MD.isSimulatedArray():
                self._fea._calResults.clearAllAttenuatorSettings = lambda: True
            self._fea._calResults.clearAllAttenuatorSettings()

    def setElevationLimit(self, elLimit=None):
        if elLimit is None:
            elLimit = self._elLimit
        self.logInfo("Setting elevation limit to %f degrees" % elLimit)
        self._mode.setElevationLimit('%f deg' % elLimit)

    def setMaximumExecutionTime(self, endTime=None):
        if endTime is None:
            endTime = self._sb.getMaximumExecutionTime()
        self.logInfo("Setting maximum execution time to %s seconds or %s minutes" % (endTime, endTime / 60.))
        self._mode.setMaximumSchedBlockTime(endTime)

    def setMaxPointingSeparation(self, maxPointingSeparation=None):
        from Observation.PointingCalTarget import PointingCalTarget
        if maxPointingSeparation is None:
            maxPointingSeparation = self._maxPointingSeparation
        self.logInfo("Setting max. pointing separation  to %f degrees" % self._maxPointingSeparation)
        PointingCalTarget.setMaximumSeparation('%f deg' % self._maxPointingSeparation)

    @performance
    def resolveQueryTargets(self, group, targetList, calibration):
        from Observation.CalibratorCatalog import getSPWList
        from Observation.CalibratorCatalog import getQueryCenter
        queryTargeList = [target for target in targetList if target.hasQuery()]
        if len(queryTargeList) == 0:
            return
        if self._noQuery:
            # For debug use
            return

        group.overwriteQueryCenterIfRequired(force=True)
        # We do not have OrderedDict...
        qsList = []
        qsToTargetMap = dict()
        for target in queryTargeList:
            qs = target.getSource()
            if qs not in qsList:
                qsToTargetMap[qs] = []
                qsList.append(qs)
            qsToTargetMap[qs].append(target)

        nQS = len(qsToTargetMap.keys())
        if nQS == 0:
            return

        if calibration == 'bandpass':
            target = queryTargeList[0]
            ra, dec = getQueryCenter(target)

            # nQS=1, nQS> 1
            sources = self._cc.getBrightGridSources(freq=100e9,
                                                    minEl=40.,
                                                    maxEl=80,
                                                    checkVisibility=True,
                                                    checkShadowing=True,
                                                    returnFieldSource=False,
                                                    nRequest=-1)
            filteredSources = []
            for source in sources:
                source.sep = source.getSeparation(ra=ra, dec=dec)
                srcName = source.getName()
                if source.sep < 2.:
                    self.logInfo("Remove %s from bandpass selection." % srcName)
                    continue
                filteredSources.append(source)

            if len(filteredSources) < nQS:
                v_ = (nQS, len(filteredSources))
                msg = "Not enough number of grid sources available"
                msg += " (While %d sources requested, only %d available)." % v_
                raise Exception(msg)

            sources = filteredSources[:nQS]
            fsList = []
            for source, target in zip(sources, queryTargeList):
                spectralSpec = target.getSpectralSpec()
                spwList = getSPWList(spectralSpec)
                freqs = np.array([spw['freq'] for spw in spwList])

                fluxes, fluxErrors = source.getFluxEstimate(freqs)
                fs = source.buildFieldSource(freqs=freqs, fluxes=fluxes)
                self._cc.copyReferencePositions(target.getSource(), fs)
                newTarget = target.getCloned(fs)
                group.replaceTarget(target, [newTarget])
        elif calibration == 'amplitude':
            # nQS=1, nQS> 1
            target = queryTargeList[0]
            ra, dec = getQueryCenter(target)
            sp = target.getSpectralSpec()

            mixerMode = getMixerMode(target).replace("solar", "MD")
            fsList = self._cc.getAmplitudeCalibrators(ra, dec, sp,
                                                      excludeSources=[],
                                                      minEl=40.,
                                                      nRequest=nQS,
                                                      checkShadowing=True,
                                                      duration=1200.,
                                                      innerRadius=2.,
                                                      mixerMode=mixerMode)
            for fs, target in zip(fsList, queryTargeList):
                self._cc.copyReferencePositions(target.getSource(), fs)
                newTarget = target.getCloned(fs)
                group.replaceTarget(target, [newTarget])
        elif calibration == 'phase':
            selectedFsList = []
            for qs in qsList:
                targets = qsToTargetMap[qs]
                nTargets = len(targets)
                # TODO...
                iTarget = 0
                target = targets[iTarget]
                mixerMode = getMixerMode(target).replace("solar", "MD")
                fsList = self._cc.getCalibrator(target, 'phase',
                                                verbose=True,
                                                nRequest=-1,
                                                innerRadius=2.,
                                                duration=1800.,
                                                mixerMode=mixerMode)
                for fs in fsList:
                    if fs in selectedFsList:
                        continue
                    # Copy reference position
                    src = target.getSource()
                    self._cc.copyReferencePositions(src, fs)
                    for target in targets:
                        newTarget = target.getCloned(fs)
                        group.replaceTarget(target, [newTarget])
                    selectedFsList.append(fs)
                    break
                else:
                    raise SBExecutionError("Cannot find enough number of calibrators")
            if len(selectedFsList) < nQS:
                raise SBExecutionError("%d phase(-like) calibrators required in total, but got only %d" % (nQS, len(selectedFsList)))

            for fs, target in zip(selectedFsList, queryTargeList):
                newTarget = target.getCloned(fs)
                group.replaceTarget(target, [newTarget])

    def propageteDetuneParameters(self, target, assocTarget):
        mixerMode = getMixerMode(target)
        ifswAttenShift, ifprAttenShift = getAttenuatorShifts(target)
        assocTarget._observingParameters["mixerMode"] = mixerMode
        assocTarget._observingParameters["ifswAttenShift"] = ifswAttenShift
        assocTarget._observingParameters["ifprAttenShift"] = ifprAttenShift

    def createAtmCalTargets(self):
        self._sb.printTargets('BEFORE createAtmCalTargets')
        self._sb.updateTargetList()
        targetAndCycleTimeList = []

        def hasATMCalibration(target):
            """
            This function should go into ObsTarget.py, in the future.
            """
            for aTarget in target.getAssociatedCalTarget():
                if isinstance(aTarget, Observation.AtmCalTarget.AtmCalTarget):
                    return True
            return False

        def register(target, cycleTime=None):
            if target.hasQuery():
                return
            band = getBand(target.getSpectralSpec())
            if band in [1, 2, 3, 4, 5, 6]:
                cycleTime = 600.0
            elif band == 7:
                cycleTime = 480.0
            elif band == 8:
                cycleTime = 420.0
            elif band in [9, 10]:
                cycleTime = 360.0
            else:
                raise SBExecutionError("Invalid band number %d [%s]" % (band))
            targetAndCycleTimeList.append((target, cycleTime))

        groupList = self._sb.getGroupList()
        for group in groupList:
            [register(target) for target in group.getBandpassCalTargetList()]
            [register(target) for target in group.getAmplitudeCalTargetList()]
            # Do not want ATM on phase calibrator
            # [register(target) for target in group.getPhaseCalTargetList()]
            [register(target) for target in group.getScienceTargetList()]

        # Construct AtmCalTarget
        atmSpectralSpecs = dict()
        for group in groupList:
            for target, cycleTime in targetAndCycleTimeList:
                if hasATMCalibration(target):
                    self.logInfo('[createAtmCalTargets] [%s] has already ATM: skip adding another one' % target)
                    continue
                atmCalTarget = self.createATMCalTarget(target,
                                                       atmSpectralSpecs,
                                                       cycleTime=cycleTime)
                self.propageteDetuneParameters(target, atmCalTarget)
                target.addAssociatedCalTarget(atmCalTarget)
                # Add ATM target to this group
                group.addTarget(atmCalTarget)
                # Tweak ATM reference position, if feasible.
                if isinstance(target, Observation.ScienceTarget.ScienceTarget):
                    continue
                atmCalTarget.tweakReferenceOffset(verbose=True,
                                                  saveOffsetToFS=True)

        self._sb.updateTargetList()
        self._sb.printTargets('AFTER createAtmCalTargets')

    def logDetuneParameters(self, target):
        mixerMode = getMixerMode(target)
        ifswAttenShift, ifprAttenShift = getAttenuatorShifts(target)
        self.logInfo("[logDetuneParameters] %6s ifsw=%4s ifpr=%4s '%s'" % (mixerMode, ifswAttenShift, ifprAttenShift, target))

    def doATMCalibrationWithZERO(self):
        """
        Perform ATM calibration with ZERO reference sub-scan enabled.
        """
        target = self._sb.getRepresentativeTarget()
        self.logInfo("[doATMCalibrationWithZERO] selecting representative target '%s'" % (target))

        sp = Observation.SSRTuning.generateAtmSpectralSpec(target.getSpectralSpec())
        tm_str = time.strftime('%Y%m%d_%H%M%S', time.gmtime())
        sp.name = "First ATM with ZERO subscan %s" % (tm_str)

        if target.getSpectralSpec().hasACACorrelatorConfiguration():
            subscanDuration = 1.92 * 3
        else:
            subscanDuration = 1.92 * 2

        # Clone spectral spec, as we do not want to re-use
        # attenuator optimisation and (correlator calibration) that will
        # be made for "normal" mixer setting.
        from Observation.AtmCalTarget import AtmCalTarget
        kwargs = dict(SubscanFieldSource=target.getSource(),
                      SpectralSpec=sp,
                      CycleTime=7200.,
                      SubscanDuration=subscanDuration,
                      IntegrationTime=0.48,
                      DataOrigin='FULL_RESOLUTION_AUTO',
                      doHotLoad=True,
                      doZero=True)
        atmTarget = AtmCalTarget(**kwargs)
        atmTarget.setOnlineProcessing(True)
        atmTarget.setUseReferencePosition(True)
        self.logInfo("[doATMCalibrationWithZERO] executing ATM cal...")
        atmTarget.execute(self._mode)
        self.listSISBias("after doATMCalibrationWithZERO()")

    def createATMCalTarget(self, target, atmSpectralSpecs, cycleTime=600.):
        def showSpectralSpec(ssp):
            self.logInfo('[showSpectralSpec] SpectralSpec name = %s' % (ssp.name))
            self.logInfo('[showSpectralSpec] Mean Freq=%s [Hz]' % (ssp.getMeanFrequency()))
            self.logInfo('[showSpectralSpec] 1st LO = %15.5f [Hz]' % (ssp.FrequencySetup.lO1Frequency.get()))
            self.logInfo('[showSpectralSpec] dopplerReference = [%s]' % (ssp.FrequencySetup.dopplerReference))
            for iB, bbs in enumerate(ssp.FrequencySetup.BaseBandSpecification):
                lo2 = bbs.lO2Frequency.get()
                cf = bbs.centerFrequency.get()
                self.logInfo('[showSpectralSpec] [%d] %s lo2=%15.5f center freq=%15.5f' % (iB, bbs.baseBandName, lo2, cf))
        from Observation.AtmCalTarget import AtmCalTarget
        tssp = target.getSpectralSpec()
        if tssp not in atmSpectralSpecs.keys():
            atmSpectralSpecs[tssp] = Observation.SSRTuning.generateAtmSpectralSpec(tssp)
            ssp = atmSpectralSpecs[tssp]
            showSpectralSpec(ssp)
        atmSpectralSpec = atmSpectralSpecs[tssp]
        self.logInfo("[createAtmCalTargets] ATM Cycle Time %6.1f [sec] [%-50s] " % (cycleTime, target))
        if target._spectralSpec.hasACACorrelatorConfiguration():
            subscanDuration = 1.92 * 3
        else:
            subscanDuration = 1.92 * 2

        if not MD.isSimulatedArray() and os.environ['LOCATION'] == 'SCO2':
            subscanDuration *= 1.5

        kwargs = dict(SubscanFieldSource=target.getSource(),
                      SpectralSpec=atmSpectralSpec,
                      CycleTime=cycleTime,
                      SubscanDuration=subscanDuration,
                      IntegrationTime=0.48,
                      DataOrigin='FULL_RESOLUTION_AUTO',
                      doHotLoad=True,
                      doZero=self._useDualMode)

        atmCalTarget = AtmCalTarget(**kwargs)
        atmCalTarget.setDoZero(False)
        solarMode = getMixerMode(target)
        atmCalTarget.mixerSetting = solarMode
        if self._useControlMD:
            atmCalTarget.setMixerMode(solarMode.lower().replace("solar", "md"))
        atmCalTarget.setOnlineProcessing(True)
        atmCalTarget._observingParameters = {}
        # Setting to use the reference position
        atmCalTarget.setUseReferencePosition(True)
        # Just for in case
        if atmCalTarget._referenceSource is None:
            self.logWarning('[WARNING] "%s" has no reference source' % (atmCalTarget))
        assert(atmCalTarget._referenceSource is not None)
        return atmCalTarget

    def prepareTargets(self):
        sb = self.getSB()
        groupList = sb.getGroupList()
        self.logInfo('The SB has %d observing groups' % (len(groupList)))
        if len(groupList) < 2:
            raise SBExecutionError("Got only %d observing group." % (len(groupList)))

        # Update spectral specs by performing doppler correction
        self.adjustDoppler()

        # Update Pointing target spectral spec to perform in-band pointing
        if hasattr(Observation.SSRTuning, "overridePointingSpectralSpecs"):
            Observation.SSRTuning.overridePointingSpectralSpecs(sb, self._array, self)

        # enable online bandpass
        for target in sb.getBandpassCalTargetList():
            self.logInfo("Online bandpass: enable online processing on target '%s'" % str(target))
            target.setOnlineProcessing(True)

        self.logInfo('Will resolve calibrator queries.')
        for group in groupList:
            self.resolveQueryTargets(group, group.getAmplitudeCalTargetList(), "amplitude")
            self.resolveQueryTargets(group, group.getBandpassCalTargetList(), "bandpass")
            targets = group.getPhaseCalTargetList() + group.getDelayCalTargetList() + group.getCheckSourceCalTargetList()
            self.resolveQueryTargets(group, targets, "phase")

            if self._noPointingQuery:
                continue
            group.associatePointingCalTargets()

        self.logInfo('Will create ATM cal targets.')
        self.createAtmCalTargets()

        self.logInfo('Will create SBRatio cal targets.')
        self.createSBRatioCalTargets()

        msg = 'Disable reference position, except for SCIENCE and ATM targets.'
        self.logInfo(msg)
        for group in groupList:
            for pTarget in group.getTargetList():
                targetList = [pTarget] + pTarget.getAssociatedCalTarget()
                for target in targetList:
                    from Observation.ScienceTarget import ScienceTarget
                    from Observation.AtmCalTarget import AtmCalTarget
                    # For solar interferometry, it is requested to take "off"
                    # point data when observing the science target.
                    if isinstance(target, ScienceTarget) or \
                       isinstance(target, AtmCalTarget):
                        useReference = True
                    else:
                        useReference = False
                    target.setUseReferencePosition(useReference)
                    values = (useReference, target)
                    self.logInfo("setUseReferencePosition=%s '%s'" % values)

    def createSBRatioCalTargets(self):
        from Observation.AtmCalTarget import AtmCalTarget
        sb = self.getSB()
        sb.updateTargetList()
        self.logInfo("Create SBRatio CalTarget for Amp and Bandpass targets only...")
        targets = sb.getAmplitudeCalTargetList() + sb.getBandpassCalTargetList() + sb.getPhaseCalTargetList()
        for target in targets:
            for assocTarget in target.getAssociatedCalTarget():
                if isinstance(assocTarget, MD.AtmCalTarget_MD) or isinstance(assocTarget, AtmCalTarget):
                    self.logInfo("Create associated SBR target for '%s'" % (assocTarget))
                    sbrTarget = assocTarget.createAssociatedSBRatioCalTarget()
                    # Disable online-processing so that this meaningless measurement
                    # does not make harm.
                    sbrTarget.setOnlineProcessing(False)
                    self.propageteDetuneParameters(assocTarget, sbrTarget)
                # sb.addTarget(sbrTarget)
        sb.updateTargetList()
        sb.printTargets('AFTER createSBRatioCalTargets')

    @performance
    def adjustDoppler(self):
        sb = self.getSB()
        groupList = sb.getGroupList()
        for group in groupList:
            sb.adjustDopplerByGroup(self._mode, group)

    def addSquareLawSetupIfNeeded(self):
        self.logInfo('[addSquareLawSetupIfNeeded] useDualMode = %s' % (self._useDualMode))
        if not self._useDualMode:
            return
        sb = self.getSB()
        for target in sb.getTargetList():
            ss = target.getSpectralSpec()
            if not hasattr(ss, 'SquareLawSetup') or ss.SquareLawSetup is None:
                self.logInfo("Adding SquareLawSetup to '%s'" % (str(ss.name)))
                ss.SquareLawSetup = CCL.APDMSchedBlock.SquareLawSetup()
                ss.SquareLawSetup.integrationDuration.set(0.016)

    def checkScienceTargetVisibility(self):
        sb = self.getSB()
        groupList = sb.getGroupList()
        for iGroup, group in enumerate(groupList):
            groupNumber = iGroup + 1
            if not self._mode.checkScienceTargetVisibility(group):
                raise SBExecutionError("No visible science target [group%d]" % (groupNumber))

    @performance
    def maximizeAttenuators(self):
        self.logInfo('Setting  IFSwitch & IFProc Max Attenuation ')
        self._fea.setMaxIFSwitchAttenuation()
        self._fea.setMaxIFProcAttenuation()

    @performance
    def createReferenceAttenuatorSettings(self):
        sb = self.getSB()
        # Get the representative science target
        target = sb.getRepresentativeTarget()
        bandNum = getBand(target.getSpectralSpec())
        mixerMode = getMixerMode(target)
        ifswAttenShift, ifprAttenShift = getAttenuatorShifts(target)

        self.logInfo('[createReferenceAttenuatorSettings] bandNum=%d mode=%s ifsw=%s ifpr=%s' % (bandNum, mixerMode, ifswAttenShift, ifprAttenShift))

        if self._ifLevel:
            self.logInfo('Change ifLevel from %.2f to %.2f' % (target.getIFLevel(), self._ifLevel))
            target.setIFLevel(self._ifLevel)
        if self._bbLevel:
            self.logInfo('Change bbLevel from %.2f to %.2f' % (target.getBBLevel(), self._bbLevel))
            target.setBBLevel(self._bbLevel)

        attenuatorSettingName = "solar_band%d_ifsw%+03d_ifpr%+03d_%s" % (bandNum, ifswAttenShift, ifprAttenShift, self._tm_str)
        if self._mode.sbe.getSessionState() and not self._mode.sbe.isFirstTime():
            self.logInfo("[createReferenceAttenuatorSettings] Using sessions: re-use previous reference attenuator setting")
        else:
            ifLevel = target.getIFLevel()
            bbLevel = target.getBBLevel()
            self.logInfo('[createReferenceAttenuatorSettings] ifLevel=%s bbLevel=%s' % (ifLevel, bbLevel))
            # Optimize attenuators, by the reference level
            if not MD.isSimulatedArray():
                # FIXME: Just for in case, should be removed in the future.
                time.sleep(2.0)

                loMode = self._mode.getLOObservingMode()
                loMode.optimizeSignalLevels(ifTargetLevel=ifLevel, bbTargetLevel=bbLevel)
                self.logInfo("Waiting for 2.6 seconds")
                # ICT-3876: see LocalOscillatorImpl.java
                time.sleep(2.6)

            if self._logPower:
                self._detune.listSBPower(self._detuneAntennaList[:3])
                self._detune.listBBPower(self._detuneAntennaList[:3])

            # Save the reference attenuator setting
            self._fea.getIFSwitchAttenuation()
            self._fea.getIFProcAttenuation()
            self.logInfo('Create attenuator setting [%s]' % attenuatorSettingName)
            if not MD.isSimulatedArray():
                # TODO: SimulatedCalResults will be updated
                self._fea.setAttenuatorSettingName(attenuatorSettingName)

        self._referenceAttenuatorSetting = dict(name=attenuatorSettingName,
                                                ifswAttenShift=ifswAttenShift,
                                                ifprAttenShift=ifprAttenShift)

        if self._mode.sbe.getSessionState():
            # UGLY HACK:
            # Use SBExecState for just checking the repetition of same SB.
            self._mode.sbe.exStateDict["sessionState"] = False

    @performance
    def buildMixerBiasDict(self, bandNum):
        self.logInfo("Setup mixer bias information [bandNum=%d" % (bandNum))
        if self._detune is None:
            self._detune = MD.MixerDetuning(self._mode, self._detuneAntennaList)
        self._detune.setupMixerBiasDict(bandNum)

    @performance
    def setMixerBias(self, bandNum, modeName):
        if self._detune is None:
            raise SBExecutionError("MixerDetuning instance not created")
        self._detune.setBias(bandNum, mixerSettingName=modeName)

    def revertMixerBias(self):
        self.logInfo("Revert mixer bias...")
        if not self._detune:
            return
        for bandNum in self._detune.getRegisteredBands():
            self.logInfo("Resetting mixer bias [band=%d]" % bandNum)
            self._detune.setBias(bandNum, mixerSettingName='normal')

    @performance
    def setSpectralSpec(self, spectralSpec):
        self.logInfo("[setSpectralSpec] Tune '%s' BEGIN _useControlMD=%s" % (spectralSpec.name, self._useControlMD))
        if not self._useControlMD:
            if self._detune is None:
                self._detune = MD.MixerDetuning(self._mode, self._detuneAntennaList)
            self._detune.setSpectralSpec(spectralSpec)
        else:
            # !! mode.setSpectrum does not do waitUntileTuned
            # !!  self._mode.setSpectrum(spectralSpec)
            if 1:
                if self._detune is None:
                    self._detune = MD.MixerDetuning(self._mode, self._detuneAntennaList)
                self._detune.setSpectralSpec(spectralSpec)
            else:
                # While performing tests on TFINT, it was found that
                # loMode.setSpectrum() does not work as expected...
                if not MD.isSimulatedArray():
                    loMode = self._mode.getLOObservingMode()
                    loMode.setSpectrum(spectralSpec)
        self.logInfo("[setSpectralSpec] Tune '%s' END" % spectralSpec.name)

    def tuneUpWithRepresentativeSetting(self):
        repTarget = self.getSB().getRepresentativeTarget()
        bandNum = getBand(repTarget.getSpectralSpec())

        # TODO: request CONTROL to add an option to
        # setSpectralSpec() to enable specification of
        # desired MD mode

        # Tune up receivers with normal mixer settings.
        self.setSpectralSpec(repTarget.getSpectralSpec())
        self.listSISBias("after first setSpectralSpec() (call)")

        # And, record mixer bias values
        self.buildMixerBiasDict(bandNum)
        # Set MD mode
        self.setMixerBias(bandNum, getMixerMode(repTarget))
        self.listSISBias("after first setMixerBias() call")

        meanFreq = repTarget.getSpectralSpec().getMeanFrequency()
        msg = "Send observing frequency to antennas, so that proper band"
        msg += " offsets be applied [freq=%8.3f GHz]" % (meanFreq / 1.0e9)
        self.logInfo(msg)

        if not MD.isSimulatedArray():
            amc = self._mode.getArrayMountController()
            self.logInfo("ICT-8436: before switching %s" % (amc.getPointingModels()))
            # Update band-offsets
            amc.setObservingFrequency(meanFreq)
            self.logInfo("ICT-8436: after  switching %s" % (amc.getPointingModels()))

    def executeTargetIfNeeded(self, target):
        isNeeded = target.isNeeded(self._mode)
        self.logInfo("'%s' isNeeded=%s" % (target, isNeeded))
        if not isNeeded:
            return
        self.executeTarget(target)

    def executeTarget(self, target):
        # Get expert parameters
        mixerMode = getMixerMode(target)
        ifswAttenShift, ifprAttenShift = getAttenuatorShifts(target)

        from Observation.PointingCalTarget import PointingCalTarget
        # HACK: First execute associated calibration targets
        for assocTarget in target.getAssociatedCalTarget():
            isNeeded = assocTarget.isNeeded(self._mode)
            self.logInfo("'%s' isNeeded=%s" % (assocTarget, isNeeded))
            if not isNeeded:
                continue
            if isinstance(assocTarget, PointingCalTarget):
                # Do not require de-tuning, just execute it
                if not MD.isSimulatedArray() and os.environ['LOCATION'] == 'SCO2':
                    continue
                if self._noPointing:
                    continue
                self.logInfo("#")
                self.logInfo("# EXEC                '%s'" % (assocTarget))
                self.logInfo("#")
                if os.environ["LOCATION"].startswith("SCO"):
                    self.logInfo("LOCATION='%s' : disalbe results application" % (os.environ["LOCATION"]))
                    assocTarget.setApply(False)

                assocTarget.execute(self._mode)
            else:
                # ATM, SBR
                self.executeTarget(assocTarget)

        self.logInfo("#")
        self.logInfo("# EXEC mode=%s ifsw=%s ifpr=%s '%s' %s %s" % (mixerMode, ifswAttenShift, ifprAttenShift, target, target.getAttenuatorOptimization(), target.getSpectralSpec().name))
        self.logInfo("#")

        # HACK: Keep the copy of assoc target list, and clear it out temporally
        assocTargetsCopy = list(target.getAssociatedCalTarget())
        target._associatedCalTarget = []

        spectralSpec = target.getSpectralSpec()
        bandNum = getBand(spectralSpec)

        if not self._useControlMD:
            # Before 2015.8, SIS mixer bias has to be changed
            # outside of target.execute() call.

            # Tune in advance
            self.setSpectralSpec(spectralSpec)
            # De-tune
            self.logInfo("set mixer bias [bandNum=%s mode=%s target=%s]" % (bandNum, mixerMode, target))
            self.setMixerBias(bandNum, mixerMode)
        else:
            target.setMixerMode(mixerMode.lower().replace("solar", "md"))

        if hasattr(target, "tweakAttenuators") and target.tweakAttenuators:
            # Explicltiy specified to tweak attenuators, using expert parameters
            tweakAttenuators = True
        elif isinstance(target, Observation.AtmCalTarget.AtmCalTarget) or isinstance(target, MD.AtmCalTarget_MD):
            # ATM cal has it's own attenuator setttings
            tweakAttenuators = False
        else:
            tweakAttenuators = True

        # Create an attenuator setting by shifting the reference one.
        if not tweakAttenuators:
            self.logInfo("'%s' do not tweak attenuator settings" % (target))
        else:
            attenuatorSettingName = "solar_band%d_ifsw%+03d_ifpr%+03d_%s" % (bandNum, ifswAttenShift, ifprAttenShift, self._tm_str)
            self.logInfo('Create attenuator setting [%s]' % attenuatorSettingName)
            if self._referenceAttenuatorSetting is None:
                raise SBExecutionError("Reference attenuator setting is empty.")

            referenceSettingName = self._referenceAttenuatorSetting["name"]
            dIFSW = ifswAttenShift - self._referenceAttenuatorSetting["ifswAttenShift"]
            dIFPR = ifprAttenShift - self._referenceAttenuatorSetting["ifprAttenShift"]

            self._fea.createShiftedAttenuatorSetting(referenceSettingName,
                                                     attenuatorSettingName,
                                                     ifswAttenShift=[dIFSW] * 4,
                                                     ifprAttenShift=[dIFPR] * 4,
                                                     antennas=self._detuneAntennaList,
                                                     overwrite=False)

            if isinstance(target, Observation.AtmCalTarget.AtmCalTarget) or isinstance(target, MD.AtmCalTarget_MD):
                target.getAttenuatorSettingName = lambda x: attenuatorSettingName if len(target._subscanList._subscanList) == 0 else ""
            else:
                target.getAttenuatorSettingName = lambda: attenuatorSettingName if len(target._subscanList._subscanList) == 0 else ""
            target.notifyAttenuatorsSet()

        if not self._useControlMD:
            # Check if all the WCAs are locked (otherwise, mixer bias will be overwritten at the beginning of next subscan)
            self._detune.checkWCALockStatus(bandNum, relock=True)

        # ICT-5552: combine series of short subscans where posible
        target.coalesceSubscans = True

        # Finally, execute it (there should be no assoc cal targets)
        assert(len(target.getAssociatedCalTarget()) == 0)
        if not MD.isSimulatedArray() and os.environ['LOCATION'] == 'SCO':
            self.logInfo('Skipping actual execution for %s' % (target))
        else:
            if isinstance(target, Observation.ScienceTarget.ScienceTarget):
                Observation.ObsTarget.ObsTarget.execute(target, self._mode, scanList=None)
            else:
                target.execute(self._mode)

        # subscanSpec = target.getSubscanSequence()
        # self.logInfo("DUMP")
        # for subscan in subscanSpec.subscans:
        #     optLevels = ["%5.1f" % level for level in subscan.optimizeAttenuators]
        #     self.logInfo("DUMP subscan %13s %5s %14s %38s %s" % (subscan.intent, getMixerMode(target), ",".join(optLevels), subscan.attenuatorSetting, target))

        self.listSISBias("after target.execute() '%s'" % (target))

        if self._logPower:
            self._detune.listSBPower(self._detuneAntennaList[:3])
            self._detune.listBBPower(self._detuneAntennaList[:3])

        # Revert associated calibration targets
        target._associatedCalTarget = assocTargetsCopy

    def listSISBias(self, msg):
        if not self._logSISBias:
            return
        sb = self.getSB()
        target = sb.getRepresentativeTarget()
        bandNum = getBand(target.getSpectralSpec())
        if bandNum <= 8:
            SBList = [1, 2]
        else:
            SBList = [1]
        if MD.isSimulatedArray():
            return
        antList = self._detuneAntennaList[:3]
        exec('from CCL.ColdCart%d import ColdCart%d as ColdCart' % (bandNum, bandNum))
        for antName in antList:
            cart = ColdCart(antName)
            for POL in [0, 1]:
                for SB in SBList:
                    func = getattr(cart, "GET_POL%d_SB%d_SIS_VOLTAGE" % (POL, SB))
                    val, timestamp = func()
                    self.logInfo("[listSISBias] %s POL%d SB%d %7.2f [mV]" % (antName, POL, SB, val * 1.0e3))

    def executeCalibrationGroup(self, group):
        sb = self.getSB()
        groupList = sb.getGroupList()
        groupNumber = groupList.index(group) + 1
        self.logInfo("Start Observing Group %d (Calibrations)" % groupNumber)

        # For simpliticy, do not consider for cylce time
        for target in group.getBandpassCalTargetList():
            self.executeTargetIfNeeded(target)

        for target in group.getAmplitudeCalTargetList():
            self.executeTargetIfNeeded(target)

    def executeScienceGroup(self, group):
        sb = self.getSB()
        groupList = sb.getGroupList()
        groupNumber = groupList.index(group) + 1
        self.logInfo("Start Observing Group %d (Calibrations)" % groupNumber)

        primaryPhaseCalTarget = self._mode.getPrimaryPhaseCal(group)
        while not group.isComplete(self._mode):
            for target in group.getPhaseCalTargetList():
                self.executeTargetIfNeeded(target)
            for target in group.getBandpassCalTargetList():
                self.executeTargetIfNeeded(target)
            for target in group.getAmplitudeCalTargetList():
                self.executeTargetIfNeeded(target)

            # Pick up a target that has not finished yet.
            sciTargetList = []
            nMapList = []
            for iT, target in enumerate(group.getScienceTargetList()):
                nMaps = target.getNumberOfMapsCompleted()
                isComplete = target.observationComplete()
                msg = "[%2d] N(map)=%5.2f complete=%s" % (iT, nMaps, isComplete)
                self.logInfo(msg)
                if isComplete:
                    continue
                sciTargetList.append(target)
                nMapList.append(int(nMaps))
            if len(sciTargetList) == 0:
                assert(False)

            target = sciTargetList[np.argmin(nMapList)]

            calList = []
            calList.extend(list(group.getPhaseCalTargetList()))
            calList.extend(list(group.getBandpassCalTargetList()))
            calList.extend(list(group.getAmplitudeCalTargetList()))

            # Execute science target
            self.executeScienceTarget(target, calList=calList,
                                      maximumTime=900.)
            if self._mode.getTimeBeforeCleanup() < 0:
                raise SBOutOfTime()
            # And, then phase cal
            self.executeTarget(primaryPhaseCalTarget)

        # Group is complete: remove the primaryPhaseCalTarget from cleanup:
        self.logInfo("Observing Group %s is completed." % groupNumber)
        self._mode.removeTargetFromCleanupList(primaryPhaseCalTarget)

    def executeScienceTarget(self, target, calList=[], maximumTime=900.):
        """
        Copy of ScienceTarget.execute...
        """
        if target.observationComplete():
            return
        # Adjust maximum execution time for science target
        maximumTime = CCL.SIConverter.toSeconds(maximumTime)
        currentElapsedTime = self._mode.getElapsedTime()
        timeBeforeCleanup = self._mode.getTimeBeforeCleanup()

        self.logInfo("[executeScienceTarget] currentElapsedTime=%s" % (currentElapsedTime))
        self.logInfo("[executeScienceTarget] timeBeforeCleanup=%s" % (timeBeforeCleanup))
        if timeBeforeCleanup:
            maximumTime = min(maximumTime, timeBeforeCleanup)
        self.logInfo("[executeScienceTarget] maximumTime=%s" % (maximumTime))
        timeToNextCal = target.getTimeUntilNextCalibrator(self._mode, calList, scanList=None)
        if timeToNextCal:
            maximumTime = min(maximumTime, timeToNextCal)
        self.logInfo("[executeScienceTarget] maximumTime=%s" % (maximumTime))
        target._maxExecutionTime = maximumTime
        self.executeTarget(target)
        # Update state info to reflect what we've executed so far
        target._currentIntegration += target._subscanList.getTimeOnSource()
        target._offsetIndex = target._offsetIndexAfterExecution

    def executeCleanupList(self):
        self.logInfo("Executing cleanup list.")
        self._mode.executeCleanupList()

    def startMovingToSun(self):
        import CCL.FieldSource
        target = self._sb.getRepresentativeTarget()
        src = target.getSource()
        self.logInfo('Move toward the science target [%s]' % (src.sourceName))
        self.logInfo('Performing setSourceAsync()')
        self._mode.setSourceAsync(src)

    def waitUntilOnSource(self):
        self.logInfo('Performing waitUntilOnSource()')
        self._mode.waitUntilOnSource()

    def endExecution(self, status):
        try:
            self._mode.resetLimits()
        except:
            self.logError("Could not reset limits")
        # if self._sessionControl:
        #     self._mode.sbe.endSBExecution()
        self._array.endExecution(status)

    def tweakExpertParameters(self):
        sb = self.getSB()
        target = sb.getRepresentativeTarget()
        bandNum = getBand(target.getSpectralSpec())
        solarMode = self._solarMode
        self.logInfo("[tweakExpertParameters] Band=%d solarMode='%s'" % (bandNum, solarMode))
        if solarMode in ['']:
            self.logInfo("[tweakExpertParameters] Do not update expert parameters")
            return

        # Dictionary of preset parameters
        presetsDictionary = dict()
        for band in [3, 6]:
            presetsDictionary[band] = dict()
            presetsDictionary[band]['normal'] = dict(ifswAttenShift=0., ifprAttenShift=0.)
        presetsDictionary[3]['solar1'] = dict(ifswAttenShift=-8., ifprAttenShift=-10.)
        presetsDictionary[3]['solar2'] = dict(ifswAttenShift=-7., ifprAttenShift=-0.)
        presetsDictionary[6]['solar1'] = dict(ifswAttenShift=-5., ifprAttenShift=-10.)
        presetsDictionary[6]['solar2'] = dict(ifswAttenShift=-8., ifprAttenShift=-0.)

        try:
            presets = presetsDictionary[bandNum][solarMode]
        except:
            msg = "Unsupported solarMode ('%s') has been specified for band%s observation." % (solarMode, bandNum)
            raise Exception(msg)

        # Update target levels for IF & BB detectors.
        self.logInfo("[tweakExpertParameters] _ifLevel '%s' -> '%s'" % (self._ifLevel, -30.))
        self._ifLevel = -30.
        self.logInfo("[tweakExpertParameters] _bbLevel '%s' -> '%s'" % (self._bbLevel, None))
        self._bbLevel = None

        self.logInfo("[tweakExpertParameters] will update/set 'mixerMode' to each target (%s)" % (solarMode))
        targets = []
        targets.extend(sb.getScienceTargetList())
        targets.extend(sb.getAmplitudeCalTargetList())
        targets.extend(sb.getBandpassCalTargetList())
        targets.extend(sb.getPhaseCalTargetList())
        [target.setObservingParameter("mixerMode", solarMode) for target in targets]

        ifsw = presets["ifswAttenShift"]
        ifpr = presets["ifprAttenShift"]
        self.logInfo("[tweakExpertParameters] will update/set the amount of attenuator shifts to each target (ifsw/ifpr=%s/%s)" % (ifsw, ifpr))
        targets = sb.getPhaseCalTargetList() + sb.getAmplitudeCalTargetList()
        for target in targets:
            for keyword in ["ifswAttenShift", "ifprAttenShift"]:
                target.setObservingParameter(keyword, presets[keyword])

try:
    # SB execution, or OSS
    array = getArray()
except:
    # manual mode
    from CCL.Global import setArrayName
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option('-a', '--arrayName')
    parser.add_option('-x', '--schedBlockName')
    parser.add_option('-s', '--session', action="store_true")
    parser.add_option('--no_pointing', action="store_true")
    opts, args = parser.parse_args()
    print "arrayName = %s" % (opts.arrayName)
    print "schedBlockName = %s" % (opts.schedBlockName)
    if opts.arrayName is None or opts.schedBlockName is None:
        parser.print_help()
        sys.exit(1)
    setArrayName(opts.arrayName)
    try:
        array = getArray()
    except:
        sys.stderr.write("Failed to get array: wrong arrayName? [%s]" % (opts.arrayName))
        sys.exit(1)
    solarObs = StandardSolarObs(array)
    try:
        solarObs.loadSB(opts.schedBlockName)
    except:
        sys.stderr.write("Failed to load SB: [%s]" % (opts.schedBlockName))
        sys.exit(1)

    if opts.no_pointing:
        solarObs.logInfo("Disable pointing calibrations...")
        solarObs._noPointing = True
        solarObs._noPointingQuery = True

    if opts.session:
        solarObs._sessionControl = 1
        solarObs.logInfo("Session Control = %s" % (solarObs._sessionControl))

else:
    array = getArray()
    solarObs = StandardSolarObs(array)
    solarObs.loadSB()

# If 'SolarMode' expert parameter is specified, then update
# every other expert parameters related to MD modes.
solarObs.tweakExpertParameters()

# None means OK. If an error occurs the status is replaced, in the catch block, by the exception.
status = None
try:
    solarObs.startExecution()
    solarObs.setElevationLimit()
    solarObs.setMaximumExecutionTime()
    solarObs.setMaxPointingSeparation()
    # Check if science targets are above the elevation limit.
    solarObs.checkScienceTargetVisibility()
    # Setup targets (resolve queries, generate ATM and SBR targets)
    solarObs.prepareTargets()
    # Enable Square-Law detector
    solarObs.addSquareLawSetupIfNeeded()
    # Maximize attenuators, just for in case
    solarObs.maximizeAttenuators()

    # Start moving to the sun. While moving, tune up the receiver
    # with science target setup, and read mixer bias.
    t0 = time.time()
    solarObs.startMovingToSun()
    solarObs.tuneUpWithRepresentativeSetting()
    solarObs.waitUntilOnSource()
    solarObs.logInfo("Took %8.3f seconds to get on source" % (time.time() - t0))

    # Adjust signal levels by looking at the Sun
    solarObs.createReferenceAttenuatorSettings()

    # Perform ATM cal scan with doZero=True (in normal mixer setting) to
    # take ZERO level values of SQLDs.
    solarObs.doATMCalibrationWithZERO()

    # Execute groups
    groupList = solarObs.getSB().getGroupList()
    solarObs.logInfo("EXECUTE GROUP1")
    solarObs.executeCalibrationGroup(groupList[0])
    for iGroup, group in enumerate(groupList[1:]):
        solarObs.logInfo("EXECUTE GROUP%d" % (iGroup + 1))
        solarObs.executeScienceGroup(group)

    solarObs.logInfo("All groups are completed. Executing cleanup list")
    solarObs.executeCleanupList()
except SBOutOfTime:
    solarObs.logWarning("Running short of time, forcing cleanup!")
    solarObs._mode._maximumTime = None  # Check
    solarObs.executeCleanupList()
except KeyboardInterrupt, ex:
    logger.logError("^C Pressed!")
    status = ex
    raise
except Exception, ex:
    solarObs.logError("Exception caught: %s" % ex)
    status = ex
    raise
finally:
    try:
        solarObs.maximizeAttenuators()
    except:
        solarObs.logError("Failed to maximise attenuators")
    try:
        if not solarObs._useControlMD:
            solarObs.revertMixerBias()
    except:
        solarObs.logError("Failed to reset mixer bias ['%s']" % (traceback.format_exc()))
    solarObs.endExecution(status)
