#!/usr/bin/env python
# ALMA - Atacama Large Millimeter Array
# (c) Associated Universities Inc., 2015
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
'''
VLBI SB-driven observation script

$Id: StandardVLBI.py 243990 2017-04-26 15:17:53Z ahirota $
'''
global sys
import sys
global os
import os
global Control
import Control
global CCL
import CCL.Global
global Observation
import Observation.SSRLogger
import Observation.SchedulingBlock
import Observation.ScanList
import Observation.CalibratorCatalog
#
# for readability all the lower level support code is either
# in the VLBICalTarget or APPSupport (APP_something() methods).
#
global APP
import Observation.APPSupport as APP


class StandardVLBI(Observation.SSRLogger.SSRLogger):
    '''
    This class executes observations using the VLBI Observing Mode.
    The initialization method creates a logger, connects to the SB,
    and calls the class execute method to do all the real work.
    '''
    def __init__(self, scriptName=None, scriptArgs=None):
        self._initTime = APP.getAcsTimestamp()
        self._array = CCL.Global.getArray() # self._arrayName)
        self._arrayName = self._array._arrayName.replace("CONTROL/", "")
        self._sb = Observation.SchedulingBlock.SchedulingBlock()
        Observation.SSRLogger.SSRLogger.__init__(self, self.__class__.__name__)
        self._logger.name = self.__class__.__name__
        self._logPrefix = ''
        self.array = None
        self.vom = None
        self._ASDM = None
        self._prepTime = self._initTime
        self._vlbiTime = self._initTime
        self._obsTimes = []
        self._doneTime = self._initTime
        self._exitTime = self._initTime
        self._status = None     # None means success
        try:                    # temp workaround
            self._scriptName = str(self._sb.ObsProcedure.obsProcScript)
            if self._scriptName is None and not scriptName is None:
                self._scriptName = scriptName
        except:
            self._scriptName = 'StandardVLBI.py'
        self._scriptArgs = scriptArgs
        self.logInfo('Standard VLBI entering execute(), '
            + str(self._scriptName))
        self.execute()

    def config(self):
        '''
        Configure the observation from the scheduling block expert parameters.
        All of these parameters are required to be present; but if they are
        unspecified (None), then we need to create suitable working values.
        This stage consists of all configuration and checking that does not
        require the VOM to exist.
        '''
        self.logInfo('Importing Expert Parameters from SB')
        # global (required) expert parameters
        self._ReferenceAntenna = self._sb.getExpertParameter('ReferenceAntenna')
        self._EfficiencyArray = self._sb.getExpertParameter('EfficiencyArray')
        self._BadAntFraction = self._sb.getExpertParameter('BadAntFraction')
        self._NoVLBI = self._sb.getExpertParameter('NoVLBI')
        self._NoPhasing = self._sb.getExpertParameter('NoPhasing')
        self._Efficiency = self._sb.getExpertParameter('Efficiency')
        self._Quality = self._sb.getExpertParameter('Quality')
        self._ScansPerAdjust= self._sb.getExpertParameter('ScansPerAdjust')
        self._EfficiencyMethod = self._sb.getExpertParameter('EfficiencyMethod')
        self._PackMode = self._sb.getExpertParameter('PackMode')
        self._StationCode = self._sb.getExpertParameter('StationCode')
        self._NChanLog2 = self._sb.getExpertParameter('NChanLog2')
        self._VLBIExpName = self._sb.getExpertParameter('VLBIExpName')
        self._ScanPrepSecs = self._sb.getExpertParameter('ScanPrepSecs')
        self._VLBISchedule = self._sb.getExpertParameter('VLBISchedule')
        self._VLBISources = self._sb.getExpertParameter('VLBISources')
        # List of 3-tuple: "scan number:vex mode:spectralSpec partId"
        self._VLBIModes = self._sb.getExpertParameter('VLBIModes')
        self._VLBIDone = self._sb.getExpertParameter('VLBIDone')
        self._sessionControl = int(self._sb.getExpertParameter('SessionControl', default=0))
        # If True, then do not perform pointing scans for bandpass and polarization
        # targets. If antenna pointing models meet the specification (2" over whole
        # sky, if I remember correctly), then there should be no major problem for
        # not having pointing scans on bandpass and polarization targets.
        self._pointingOnce = int(self._sb.getExpertParameter("PointingOnce", default=0))
        maxPointingSeparation = float(self._sb.getExpertParameter('MaxPointingSeparation', default=45.))
        maxSep = "%f deg" % maxPointingSeparation
        from Observation.PointingCalTarget import PointingCalTarget
        PointingCalTarget.setMaximumSeparation(maxSep)

        # For last-minute change of expert parameters.
        pFile = self._sb.getExpertParameter("LastMinuteChangeFile", default="").strip()
        self.logInfo("LastMinuteChangeFile = '%s'" % pFile)
        if pFile != "":
            APP.APP_UpdateExpertParameters(self, pFile)

        # Tweak some parameters for a regression test.
        if int(self._sb.getExpertParameter("Regression", default=0)) == 1:
            import Observation.APPRegression as APPRegression
            # 'VLBISchedule', 'VLBISources', and 'VLBIDone' will be updated.
            APPRegression.APP_UpdateParameters(self)

        groupList = self._sb.getGroupList()
        if len(groupList) < 2:
            msg = "VLBI Scheduling Block requires two observing groups, " + \
                  "but only %d configured." % len(groupList)
            raise Exception(msg)

        # ICT-9846
        if int(self._sb.getExpertParameter("LazyPointingQuery", default=1)):
            self.logInfo("Enable lazy evaluation of pointing query")
            for group in groupList:
                group.enableLazyPointingQuery(True)

        APP.APP_SBConfig(self)
        APP.APP_SelfInit(self)

    def prepare(self):
        '''
        This method creates the VOM, begins execution and performs
        any required initial calibration or other activities.
        '''
        self._prepTime = APP.getAcsTimestamp()
        self.logInfo('Observation Preparations')
        # pre-execution setup and checks
        self.array = self._array
        APP.APP_AntennaConfig(self)
        APP.APP_ApsModeConfig(self)
        # Configure spectral spec
        APP.APP_SpectralCheck(self)
        # Make sure schedule and targets match
        APP.APP_TargetCheck(self)
        # Configure _VLBIScanList
        APP.APP_ScheduleCheck(self)
        # Report on PIC and other VLBI readiness
        APP.APP_PICStatus(self)
        APP.APP_FinalRunTimeChecks(self)
        self.array.setAppSumAntenna(True)
        self.array.beginExecution()
        # post-execution configuration
        try: # ICT-3188 workaround
            self._ASDM = self._array.getExecBlockUID()
        except:
            self._ASDM = self._array._array.getExecBlock().entityId
        self.logInfo("Beginning Execution, uid is %s" % (self._ASDM))
        self.reportShiftLogIfRequired()
        self.vom = self.array.getVLBIObservingMode()
        # Configure maximum time for single SB execution
        endTime = self._sb.getMaximumExecutionTime()
        self.vom.setMaximumSchedBlockTime(endTime)
        # Add some SSR
        self.addSSRModetoVOM()
        # SSR: Identify VLBI targets that needs to be assigned with OBSERVE_TARGET intent.
        self.identifyScienceScans()
        # SSR: If spectral specs for ALMA calibrations do not match with VLBI specs, replace them.
        self.checkCalibrationSpectralSpecs()
        # SSR: Resolve calibrator queries
        self.expandSourceQueries()
        # SSR: Add CALIBRATE_FLUX itent to bandpass targets, if amp target does not exist.
        self.tweakFluxCalIntent()
        # SSR: Add generated VLBI targets (_VLBIScanList) to group2
        self.addVLBITargetsToGroup2()
        # SSR: Create ATM cal targets
        self.createAtmCalTargets()
        # SSR: Select pointing sources
        self.selectPointingTargets()
        # SSR: Ugly sessions...
        if self._sessionControl:
            self.vom.sbe.recycleAttenSettings()
            self._sb.setTargetsId()
            self.vom.sbe.recycleFieldSources()
        APP.APP_TelCalConfig(self)
        APP.APP_VOMConfig(self)

    def identifyScienceScans(self):
        group = self.getObservingGroup(2)
        for vlbiTarget in self._VLBIScanList:
            vlbiSS = vlbiTarget.getSpectralSpec()
            for target in group.getScienceTargetList():
                if target.getSpectralSpec() == vlbiSS and target.getSource() == vlbiTarget.getSource():
                    self.logInfo("Mark '%s' as science source (matched with '%s')" % (vlbiTarget, target))
                    vlbiTarget.isScienceSource = True
                    break
                else:
                    self.logInfo("VLBI '%s' is not Science '%s'" % (vlbiTarget, target))

        if len([vT for vT in self._VLBIScanList if vT.isScienceSource]) > 0:
            return
        self.logInfo("None of VLBI targets matched with science targets.")

    def tweakFluxCalIntent(self):
        groupList = self._sb.getGroupList()
        if sum([len(group.getAmplitudeCalTargetList()) for group in groupList]) > 0:
            return
        self.logInfo("No amplitude target defined: Will CALIBRATE_FLUX intent to bandpass target")
        for group in groupList:
            for target in group.getBandpassCalTargetList():
                target.setFluxCalReduction(True)

    def reportShiftLogIfRequired(self):
        self.logInfo("_scriptName=%s" % self._scriptName)
        self.logInfo("_scriptArgs=%s" % self._scriptArgs)
        if self._scriptName and self._scriptArgs:
            self.shiftlog(self._ASDM, self._scriptName, scriptArgs=self._scriptArgs)

    def addSSRModetoVOM(self):
        '''
        An ugly hack shared among SSR observation scripts.
        '''
        import Observation.SBExecState
        from Observation.SBExecutionMode import SBExecutionMode
        from Observation.Global import addClassToMode
        addClassToMode(self.vom, SBExecutionMode)
        sbe = Observation.SBExecState.SBExecState(self._array, self._sb)
        self.vom.addSBExecState(sbe)
        if self._sessionControl:
            self.vom.sbe.startSession()

    def checkCalibrationSpectralSpecs(self):
        vlbiSpecs = [vlbiTarget.getSpectralSpec() for vlbiTarget in self._VLBIScanList]
        vlbiSpecs = list(set(vlbiSpecs))
        if len(vlbiSpecs) > 1:
            raise Exception("Multiple spectral specs for VLBI targets are not supported.")
        vlbiSS = vlbiSpecs[0]
        targets = []
        for group in self._sb.getGroupList():
            targets.extend(group.getAmplitudeCalTargetList())
            targets.extend(group.getBandpassCalTargetList())
            targets.extend(group.getPhaseCalTargetList())
            targets.extend(group.getPolarizationCalTargetList())
            #targets.extend(group.getCheckSourceCalTargetList())
            targets.extend(group.getDelayCalTargetList())

        for target in targets:
            ss = target.getSpectralSpec()
            if ss not in vlbiSpecs:
                self.logWarning("Spectral spec for '%s' does not match for any of VLBI spec(s)" % target)
                target.setSpectralSpec(vlbiSS)

    def getObservingGroup(self, groupNumber):
        groups = self._sb.getGroupList()
        return groups[groupNumber - 1]

    def addVLBITargetsToGroup2(self):
        group = self.getObservingGroup(2)
        for vlbiTarget in self._VLBIScanList:
            group.addTarget(vlbiTarget)

    def createAtmCalTargets(self):

        def register(targetCycleTimeDict, target, cycleTime=None):
            """
            Helper function for adding ATM cal to the target
            """
            if target.hasQuery():
                # Queries should have already resolved at this stage.
                return
            tssp = target.getSpectralSpec()
            band = self._sb.getBand(target)
            if cycleTime is None:
                if band in [1, 2, 3, 4, 5, 6]:
                    cycleTime = 600.0
                elif band == 7:
                    cycleTime = 480.0
                elif band == 8:
                    cycleTime = 420.0
                elif band in [9, 10]:
                    cycleTime = 360.0
                else:
                    raise Exception("Invalid band number %d [%s]" % (band))
            if target in targetCycleTimeDict:
                self.logInfo('[createAtmCalTargets] [%-60s] cycleTime=%6.1f oldCycleTime=%6.1f' %
                            (target, cycleTime, targetCycleTimeDict[target]))
            else:
                self.logInfo('[createAtmCalTargets] [%-60s] cycleTime=%6.1f' % (target, cycleTime))
            targetCycleTimeDict[target] = cycleTime

        self.logInfo("[createAtmCalTargets] Create ATM targets for ALMA calibrations scans.")

        atmSSCache = dict()
        fsToAtmTargetDict = dict()
        groupList = self._sb.getGroupList()
        for group in groupList:
            targetCycleTimeDict = dict()
            # Do not add ATM to phase calibrator or CheckSourceTargets, &c
            # but do make sure ATM ends up on VLBI, and Pol, Amp & BP.
            if group.groupIndex == 2:
                [register(targetCycleTimeDict, target) for target in self._VLBIScanList]
            [register(targetCycleTimeDict, target) for target in group.getPolarizationCalTargetList()]
            [register(targetCycleTimeDict, target) for target in group.getAmplitudeCalTargetList()]
            [register(targetCycleTimeDict, target) for target in group.getBandpassCalTargetList()]
            # If 'forceATM' is checked on OT, set cycle time to 6 seconds.
            [register(targetCycleTimeDict, target, cycleTime=6.) for target in group.getTargetList() if target.forceAtm]
            # Construct ATM cal target instances
            for target, cycleTime in targetCycleTimeDict.items():
                isVLBITarget = target in self._VLBIScanList
                fs = target.getSource()

                # Share same ATM target instance among VLBI scans with a same source, so that
                # ATM scans will follow cycle time rule.
                shareATM = isVLBITarget and (fs in fsToAtmTargetDict)
                if shareATM:
                    atmTarget = fsToAtmTargetDict[fs]
                else:
                    atmTarget = group.constructATMTarget(target, cycleTime,
                                                         atmSpectralSpecCacheDict=atmSSCache)
                    if atmTarget is None:
                        continue
                    # ICT-7734: tweak ATM reference position
                    atmTarget.tweakReferenceOffset(verbose=False, saveOffsetToFS=True)
                # once is enough
                if not atmTarget in target.getAssociatedCalTarget():
                    target.addAssociatedCalTarget(atmTarget)
                if isVLBITarget:
                    fsToAtmTargetDict[fs] = atmTarget

            group.printTargets("Added ATM to normal targets")

        # Print target list
        for group in groupList:
            group.printTargets("AFTER createAtmCalTargets")

    def expandSourceQueries(self):
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing', useSSRLogger=True)
        for group in self._sb.getGroupList():
            # Perform doppler correction for spectral specs
            self._sb.adjustDopplerByGroup(self.vom, group)
            # Resolve target queries
            cc.resolveGroupQueries(group)

    def selectPointingTargets(self):
        groupList = self._sb.getGroupList()

        # Identify observing group that has pointing target.
        # (In the future, we have to make a change request
        # to OT team so that pointing target will be populated
        # under both group1 and group2.)
        groupWithPnt = None
        for group in groupList:
            hasPointing = len(group.getPointingCalTargetList()) > 0
            self.logInfo("Group%d hasPointing=%s" % (group.groupIndex, hasPointing))
            if hasPointing:
                groupWithPnt = group
                break
        else:
            self.logInfo("No pointing target defined: will not perform pointing scans")
            return

        # Cache dictionary: Key is FieldSource and value is PointingCalTarget
        fsToPntTargetDict = dict()
        # Identify which targets to add pointing target
        targets = []
        if self._pointingOnce:
            # Just add pointing target to VLBI targets only.
            targets.extend(self._VLBIScanList)
        else:
            # Add pointing to every targets.
            for group in groupList:
                targets.extend(group.getPhaseCalTargetList())
                targets.extend(group.getPolarizationCalTargetList())
                targets.extend(group.getAmplitudeCalTargetList())
                targets.extend(group.getBandpassCalTargetList())
                #targets.extend(group.getCheckSourceCalTargetList())
                targets.extend(group.getDelayCalTargetList())
            targets.extend(self._VLBIScanList)

        for target in targets:
            pntTarget = groupWithPnt.selectPointingCalTarget(target, cacheDict=fsToPntTargetDict)
            if pntTarget is None:
                self.logInfo("[selectPointingTargets] no pointing source for '%s'" % (target))
                continue
            self.logInfo("[selectPointingTargets] selected %s for '%s'" % (pntTarget.getSource().sourceName, target))
            pntTarget.setUseReferencePosition(False)
            target.addAssociatedCalTarget(pntTarget, 0)
            fsToPntTargetDict[target.getSource()] = pntTarget
            # Disable TelCal results application on SCO simulated STEs...
            if os.environ["LOCATION"].startswith("SCO"):
                self.logInfo("LOCATION='%s' : disable results application" % (os.environ["LOCATION"]))
                pntTarget.setApply(False)

    def getEntryMargin(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return (vlbiTarget.CorrStartTime - curTime) / 10000000.0

    def getTimeAndMargin(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return curTime, (vlbiTarget.CorrStartTime - curTime) / 10000000.0

    def getTimeAndMarginExit(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return curTime, (vlbiTarget.CorrEndTime - curTime) / 10000000.0

    def estimateDuration(self, target):
        interSubscanLatency = 1.5
        # FIXME/TODO: update and these values, if required
        # apply pointing results => 8 seconds?
        # attenuatorOptimization = 0.
        # correlatorCalibration = 0.
        # tSetPointingDirection = 4.
        # tCorrCalibration = 4.
        # tTune = 23.
        # tInterSubscan = 1.5
        # tTelCal = 8

        target._populateSubscanList()
        subscanSpec = target.getSubscanSequence()
        subscanDurations = [subscan.duration for subscan in subscanSpec.subscans]

        duration = sum(subscanDurations)
        duration += len(subscanDurations) * interSubscanLatency

        # Add associated calibration targets
        for aTarget in target.getAssociatedCalTarget():
            isNeeded = aTarget.isNeeded(self.vom)
            if not isNeeded:
                continue
            className = aTarget.__class__.__name__
            if className == 'AtmCalTarget':
                duration += 90.
            elif className == 'PointingCalTarget':
                duration += 210.
            else:
                self.logWarning("[estimateDuration] unknown calibration type '%s'" % (aTarget))
        return duration

    def checkCalibrationTargets(self, group):
        targets = []
        targets.extend(group.getPhaseCalTargetList())
        targets.extend(group.getPolarizationCalTargetList())
        targets.extend(group.getAmplitudeCalTargetList())
        targets.extend(group.getBandpassCalTargetList())
        # targets.extend(group.getCheckSourceCalTargetList())
        targets.extend(group.getDelayCalTargetList())
        targetAndDurationList = []
        for target in targets:
            isObservable = target.isObservable(self.vom, duration=600)
            isNeeded = target.isNeeded(self.vom)
            if not isObservable or not isNeeded:
                continue
            duration = self.estimateDuration(target)
            self.logInfo("[checkCalibrationTargets] estimated duration=%5.1f sec '%s' integration time=%5.1f sec" % (duration, target, target.getIntegrationTime()))
            targetAndDurationList.append((target, duration))
        return targetAndDurationList

    def doPointingIfRequired(self, target):
        '''
        This method does pointing for the target (if required)
        '''
        pntTargets = []
        for aTarget in target.getAssociatedCalTarget():
            if aTarget.__class__.__name__ == 'PointingCalTarget':
                pntTargets.append(aTarget)
        # There should only 1 (or 0) pointing target associated.
        if len(pntTargets) == 0:
            return
        pntTarget = pntTargets[0]
        if not pntTarget.isNeeded(self.vom):
            return

        # disable phasing and restore standard delay handling
        self.logInfo('AppMode type "calalma"')
        self.vom.setAppScanParameters(
            self._appModeSeq['calalma'], self._PackMode)

        self.logInfo("Perform pointing ... '%s'" % (pntTarget))
        pntTime, entry = self.getTimeAndMargin(target)
        pntTarget.execute(self.vom)
        curTime, exit = self.getTimeAndMargin(target)
        self.describeCal('<point>', pntTarget, pntTime, curTime, entry, exit)

    def executeCalTarget(self, target, action="<calib>", force=False):
        '''
        Execute ALMA specific calibration scan. Rather than just calling
        target.execute(), this method used executeAssociatedCalTargetList()
        so that timing information targets can be recorded.
        '''
        isObservable = target.isObservable(self.vom, duration=600)
        isNeeded = target.isNeeded(self.vom) or force
        if not isObservable or not isNeeded:
            self.logWarning("isObservable=%d isNeeded=%d skip '%s'" % (isObservable, isNeeded, target))
            return
        # Execute associated calibration targets (pointing or ATM).
        self.executeAssociatedCalTargetList(target)
        # Temporily evacuate associated calibration targets (just in case).
        assocTargetsBackup = list(target.getAssociatedCalTarget())
        target.clearAssociatedCalTarget()

        # disable phasing and restore standard delay handling
        self.logInfo('AppMode type "calalma"')
        self.vom.setAppScanParameters(
            self._appModeSeq['calalma'], self._PackMode)

        # Execute the calibration
        eTime, entry = self.marginHelper(self._FirstVLBITarget)
        target.execute(self.vom)
        cTime, exit = self.marginHelper(self._FirstVLBITarget)
        self.describeCal(action, target, eTime, cTime, entry, exit)
        # Restore associated calibration targets
        [target.addAssociatedCalTarget(aT) for aT in assocTargetsBackup]

    def marginHelper(self, target):
        from Observation.VLBICalTarget import VLBICalTarget
        if isinstance(target, VLBICalTarget):
            time, margin = self.getTimeAndMargin(target)
        elif self._FirstVLBITarget:
            time, margin = self.getTimeAndMargin(self._FirstVLBITarget)
        else:
            time, margin = APP.getAcsTimestamp(), 0.
        return time, margin

    def executeAssociatedCalTargetList(self, target, action="<assoc>"):
        for aTarget in target.getAssociatedCalTarget():
            className = aTarget.__class__.__name__
            isNeeded = aTarget.isNeeded(self.vom)
            self.logInfo("[assoc] isNeeded=%s '%s'" % (isNeeded, aTarget))
            if not isNeeded:
                continue
            if className == "PointingCalTarget":
                APP.APP_TestingPointing(self)
            elif className == "AtmCalTarget":
                APP.APP_TestingAtm(self)
            else:
                self.logWarning("Unsupported target for associated calibrations: '%s'" % (aTarget))

            # disable phasing and restore standard delay handling
            self.logInfo('AppMode type "calalma"')
            self.vom.setAppScanParameters(
                self._appModeSeq['calalma'], self._PackMode)

            eTime, entry = self.marginHelper(target)
            aTarget.execute(self.vom)
            cTime, exit = self.marginHelper(target)
            self.describeCal(action, aTarget, eTime, cTime, entry, exit)

    def calibratePriorToVLBILoop(self):
        """
        Execute group1 targets (should be ALMA specific calibrations)
        """
        if self._pointingOnce:
            # If _pointingOnce=True, then Perform pointing scan only for
            # the first VLBI source.
            for vlbiTarget in self._VLBIScanList:
                if self.getEntryMargin(vlbiTarget) < 90:    # approx
                    continue
                self._FirstVLBITarget = vlbiTarget
                APP.APP_TestingPointing(self)
                self.doPointingIfRequired(vlbiTarget)
                break
        else:
            # For margin calculations within the following, we define:
            self._FirstVLBITarget = self._VLBIScanList[0]

        # Perform as many as possible group1 targets before the first
        # VLBI scan.
        self.calibratePriorToScan(self._FirstVLBITarget, groupIndex=1)

    def calibratePriorToScan(self, vlbiTarget, groupIndex=2):
        '''
        This method is called to arrange calibrations prior to the next (VLBI)
        scan.  The most important thing is to be done (with margin) before
        the correlator must be running the scan.  The first scan is special
        in that various one-time calibrations happen before VLBI begins.
        The VLBI scan is presented to provide information about the scan,
        and most importantly, when it is scheduled to start.
        '''
        # note next target for diagnostics
        self._FirstVLBITarget = vlbiTarget

        group = self.getObservingGroup(groupIndex)
        # If there is a margin, perform some ALMA calibration scans.
        while 1:
            margin = self.getEntryMargin(vlbiTarget)
            self.logInfo('[calibratePriorToScan] Waiting for %s at %s [margin=%.1f sec]' % (vlbiTarget.SourceName, vlbiTarget.RecVexTime, margin))
            if margin < 90:
                break

            # Pick up an appropriate target to execute
            targetToExecute = None
            for target, duration in self.checkCalibrationTargets(group):
                if duration > margin:
                    continue
                targetToExecute = target

            if targetToExecute is None:
                self.logInfo('[calibratePriorToScan] Has a margin of %.1f sec, but there is nothing else to slip in' % (margin))
                break

            self.logInfo('[calibratePriorToScan] picked up "%s"' % targetToExecute)
            self.executeCalTarget(targetToExecute, action='<calib>')
        if groupIndex == 1:
            return

        # Just for in case, check exit margin for the next VLBI scan.
        # If it is too late, it does not make sense to perform calibrations.
        curTime, exitMargin = self.getTimeAndMarginExit(vlbiTarget)
        self.logInfo("[calibratePriorToScan] exit margin for next VLBI scan: %.1f [seconds]" % (exitMargin))
        if exitMargin < 0.:
            self.logInfo("[calibratePriorToScan] It is too late and thus will not perform any calibration for '%s'" % vlbiTarget)
            return

        try:
            from Observation.CalibratorSource import getTimeToRise
            src = vlbiTarget.getSource()
            timeToRise_ = getTimeToRise(src, elLimit=20.)
            msg = "Time to rise = %.1f [seconds] ('%s')" % (timeToRise_, vlbiTarget)
            self.logInfo("[calibratePriorToScan] %s" % msg)
            # TODO: if timeToRise_ is not a huge positive number, wait for it.
        except:
            # Just for in case...
            import traceback
            self.logInfo("%s" % (traceback.format_exc()))

        # Then do associated calibrations for VLBI target (pointing or ATM).
        self.executeAssociatedCalTargetList(vlbiTarget)

    def calibrateAfterScans(self):
        '''
        If we are not yet at the appointed time (VLBIDone) and if there
        are calibrations that remain to be done, then do them now.
        '''
        self.logInfo('Expecting to Finish by VLBIDone: ' + self._ThisVLBIDone)
        curTime = APP.getAcsTimestamp()
        if curTime > self._AcsDoneTime:
            self.logInfo('No time left for residual calibrations')
            return

        self.logInfo('Will perform final polarization calibration')
        group = self.getObservingGroup(2)
        self._FirstVLBITarget = None
        for target in group.getPolarizationCalTargetList():
            self.executeCalTarget(target, action="<after>", force=True)

    def describeCal(self, action, target, atime, curTime, entry, exit):
        '''
        Enquiring minds will want to know how we did
        '''
        # if action == '':
        #     className = target.__class__.__name__
        #     action = className.replace("CalTarget", "").lower()
        #     action = "<%s>" % (action.replace("ing", ""))
        dwell = int(float(curTime - atime + 0.5) / 10000000.0)
        outcome = '%s (%ds)' % (target, dwell)
        self._obsTimes.append((curTime, entry, exit, action, outcome))

    def describeScan(self, vlbiScan, curTime, dwell, entry, exit):
        '''
        Enquiring minds will want to know how we did
        '''
        if vlbiScan.ActivePhase:
            vmode = 'Active  ' + vlbiScan.ApsMode
        else:
            vmode = 'Passive ' + vlbiScan.ApsMode
        if exit == None:
            outcome = 'skip %s at %s (0s/0s/0.0s) (%s)' % (
                vlbiScan.SourceName, vlbiScan.RecVexTime, vmode)
            self.logInfo('%s [%.3f]' % (outcome, -entry))
        else:
            APP.APP_ReportScanStatus(self)
            did = (vlbiScan.CorrEndTime - vlbiScan.CorrStartTime) / 10000000.0
            outcome = 'vlbi %s at %s (%ds/%ds/%.1fs) (%s)' % (
                vlbiScan.SourceName, vlbiScan.RecVexTime,
                dwell, int(vlbiScan.ScanSeconds), did, vmode)
            self.logInfo('%s [%.3f %.3f]' % (outcome, entry, exit))
        self._obsTimes.append(
            (curTime, entry, exit, vlbiScan.ScanName, outcome))

    def performVlbiScan(self, vlbiScan):
        '''
        This method is called to execute the provided VLBI scan.
        '''
        self.logInfo('performing  %s at %s, AppMode type %s' % (
            vlbiScan.SourceName, vlbiScan.RecVexTime, vlbiScan.ApsMode))
        curTime, entryMargin = self.getTimeAndMargin(vlbiScan)
        dwell = curTime
        exitMargin = None
        if (self._lateMargin < 0):
            vlbiScan.reduceSubscanRepeats(
                self._entryMargin, entryMargin, self._lateMargin)
            entryMargin = self.getEntryMargin(vlbiScan)
        # VEX might consider it observable, but ALMA rules here.
        isObservable = vlbiScan.isObservable(self.vom,
            duration=vlbiScan.ScanSeconds)
        if not isObservable:
            self.logWarning('NOT OBSERVABLE (entry margin %.3f)' % entryMargin)
            entryMargin = 0.0
        APP.APP_TestingScan(self, vlbiScan, 0)
        # Temporily evacuate associated calibration targets (pointing or ATM),
        # so that they will not intrude into a VLBI scan.
        assocTargetsBackup = list(vlbiScan.getAssociatedCalTarget())
        vlbiScan.clearAssociatedCalTarget()
        if entryMargin > self._entryMargin:
            vlbiScan.setWVRReduction()
            vlbiScan.checkRates(self._numBasebands, self._numberAntennas)
            self.vom.setAppScanParameters(
                self._appModeSeq[vlbiScan.ApsMode], self._PackMode)
            scanList = Observation.ScanList.ScanList()
            for xx in range(vlbiScan.CorrSubscanRepeats):
                vlbiScan.execute(self.vom, scanList)
            APP.APP_TestingScan(self, vlbiScan, 1)
            scanList.execute(self.vom, startTime=vlbiScan.CorrStartTime)
            curTime, exitMargin = self.getTimeAndMarginExit(vlbiScan)
            dwell = int(float(curTime - dwell + 0.5) / 10000000.0)
            vlbiScan.restoreRates()
        self.describeScan(vlbiScan, curTime, dwell, entryMargin, exitMargin)
        # Restore associated calibration targets
        [vlbiScan.addAssociatedCalTarget(aT) for aT in assocTargetsBackup]

    def observe(self):
        '''
        This method runs the full set of VLBI observations.
        '''
        self._vlbiTime = APP.getAcsTimestamp()
        self.logInfo('Configuring PICs; this may take a minute')
        APP.APP_PICSetup(self)
        self.logInfo('Delivering schedule to the recorders')
        APP.APP_RecorderSetup(self)
        self.logInfo("Perform group 1 calibration targets prior to enter the VLBI observing loop")
        self.calibratePriorToVLBILoop()
        self.logInfo('Entering VLBI Observing Loop')
        for vs in self._VLBIScanList:
            self.logInfo('working     %s at %s' % (
                vs.SourceName, vs.RecVexTime))
            self._NextFieldSource = vs.SubscanFieldSource
            # with active phasing we can afford the calibrations
            # with passive phasing we need to leave the TFBs undisturbed
            if vs.ActivePhase:
                self.calibratePriorToScan(vs)
            self.performVlbiScan(vs)
        self.calibrateAfterScans()
        APP.APP_PICStatus(self)

    def teardown(self):
        '''
        Following the observation, this shuts down, destroying the
        VOM and ending the execution block.
        '''
        self._doneTime = APP.getAcsTimestamp()
        self.logInfo('Post-observation cleanup')
        if self.vom:
            try:
                APP.APP_RecorderAbort(self)
            except Exception, ex:
                self.logException("Unable to abort recorder schedule", ex)

            if self._sessionControl and hasattr(self.vom, "sbe"):
                self.vom.sbe.endSBExecution()

        if self.array:
            try:
                self.array.endExecution(self._status)
                self.array.setAppSumAntenna(False)
            except Exception, ex:
                self.logException("Unable to endExecution", ex)
        self._exitTime = APP.getAcsTimestamp()
        APP.APP_TimingSummary(self)

    def execute(self):
        '''
        This method executes the complete observation, start to finish.
        '''
        where = 'nowhere'
        try:
            where = 'config()'
            self.config()
            where = 'prepare()'
            self.prepare()
            where = 'observe()'
            self.observe()
        except Exception, ex:
            import traceback
            self.logInfo("%s" % (traceback.format_exc()))
            self.logException("Exception in " + where, ex)
            self._status = ex
        except KeyboardInterrupt:
            self.logError("^C interrupt...shutting down")
        finally:
            self.teardown()


def manualModeExecution():
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option('-a', '--arrayName')
    parser.add_option('-x', '--schedBlockName')
    opts, args = parser.parse_args()
    if opts.arrayName is None or opts.schedBlockName is None:
        parser.print_help()
        sys.exit(1)
    CCL.Global.setArrayName(opts.arrayName)
    array = CCL.Global.getArray(opts.arrayName)
    CCL.Global.loadSB(opts.schedBlockName)
    vlbi = StandardVLBI(scriptName="StandardVLBI.py", scriptArgs=" ".join(args))


if __name__ == "__builtin__":
    # Interactive array execution
    vlbi = StandardVLBI()
elif __name__ == "__main__":
    # Manual array execution
    manualModeExecution()
elif "ObservingModeSimulator" in __name__:
    # ObservingScriptSimulator execution
    vlbi = StandardVLBI()
#
# eof
#
