#!/usr/bin/env python
# ALMA - Atacama Large Millimeter Array
# (c) Associated Universities Inc., 2015..2019
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
'''
VLBI SB-driven observation script
'''
from __future__ import absolute_import
global sys
import sys
global os
import os
global Control
import Control
global CCL
import CCL.Global
global Observation
import Observation.SSRLogger
import Observation.SchedulingBlock
import Observation.ScanList
import Observation.CalibratorCatalog
#
# for readability all the lower level support code is either
# in the VLBICalTarget or APPSupport (APP_something() methods).
#
global APP
import Observation.APPSupport as APP
global APS_Delays
from Observation.APPDelays import APS_Delays

import Observation.APSAnnotate

class StandardVLBI(Observation.SSRLogger.SSRLogger):
    '''
    This class executes observations using the VLBI Observing Mode.
    The initialization method creates a logger, connects to the SB,
    and calls the class execute method to do all the real work.
    scriptName and scriptArgs exist for the shiftlog entries.
    '''
    def __init__(self, scriptName=None, scriptArgs=None):
        self._initTime = APP.getAcsTimestamp()
        self._array = CCL.Global.getArray() # self._arrayName)
        self._arrayName = self._array._arrayName.replace("CONTROL/", "")
        self._sb = Observation.SchedulingBlock.SchedulingBlock()
        Observation.SSRLogger.SSRLogger.__init__(self, self.__class__.__name__)
        self.ano = Observation.APSAnnotate.APS_Annotate
        self._logger.name = self.__class__.__name__
        self._logPrefix = ''
        self.array = None
        self.vom = None
        self._ASDM = None
        self.where = 'init()'
        self.tcparams = None
        self.rbands = set()
        self.apscalibs = dict()
        self.apsdelays = None
        self.apsisflat = [False,False,False,False]
        self._SpecLineList = list()
        self._prepTime = self._initTime
        self._vlbiTime = self._initTime
        self._obsTimes = []
        self._doneTime = self._initTime
        self._exitTime = self._initTime
        self._scriptArgs = scriptArgs
        self._status = None     # None means success
        try:                    # temp workaround
            self._scriptName = str(self._sb.ObsProcedure.obsProcScript)
            if self._scriptName is None and not scriptName is None:
                self._scriptName = scriptName
        except:
            self._scriptName = 'StandardVLBI.py'
        self.logInfo('Standard VLBI entering execute(), '
            + str(self._scriptName) + ' at ' + os.environ["LOCATION"])
        self.ano(self, 'v', self._scriptName + ' execution starts')
        self.execute()

    def config(self):
        '''
        Configure the observation from the scheduling block expert parameters.
        All of these parameters are required to be present; but if they are
        unspecified (None), then we need to create suitable working values.
        This stage consists of all configuration and checking that does not
        require the VOM to exist.
        '''
        self.logInfo('Importing Expert Parameters from SB')
        # global (required) expert parameters
        self._ReferenceAntenna = self._sb.getExpertParameter('ReferenceAntenna')
        self._EfficiencyArray = self._sb.getExpertParameter('EfficiencyArray')
        self._BadAntFraction = self._sb.getExpertParameter('BadAntFraction')
        self._NoVLBI = self._sb.getExpertParameter('NoVLBI')
        self._NoPhasing = self._sb.getExpertParameter('NoPhasing')
        self._Efficiency = self._sb.getExpertParameter('Efficiency')
        self._Quality = self._sb.getExpertParameter('Quality')
        self._ScansPerAdjust= self._sb.getExpertParameter('ScansPerAdjust')
        self._EfficiencyMethod = self._sb.getExpertParameter('EfficiencyMethod')
        self._PackMode = self._sb.getExpertParameter('PackMode')
        self._StationCode = self._sb.getExpertParameter('StationCode')
        self._NChanLog2 = self._sb.getExpertParameter('NChanLog2')
        self._VLBIExpName = self._sb.getExpertParameter('VLBIExpName')
        self._ScanPrepSecs = self._sb.getExpertParameter('ScanPrepSecs')
        self._VLBISchedule = self._sb.getExpertParameter('VLBISchedule')
        self._VLBISources = self._sb.getExpertParameter('VLBISources')
        # List of 3-tuple: "scan number:vex mode:spectralSpec partId"
        self._VLBIModes = self._sb.getExpertParameter('VLBIModes')
        self._VLBIDone = self._sb.getExpertParameter('VLBIDone')
        self._sessionControl = int(self._sb.getExpertParameter('SessionControl', default=0))
        self._RequireVLBIDone = int(self._sb.getExpertParameter('RequireVLBIDone', default=1))
        self._VLBIOptional = int(self._sb.getExpertParameter('VLBIOptional', default=0))
        self._VLBICalMargin = float(self._sb.getExpertParameter('VLBICalMargin', default=90.))
        self._VLBIPriorMargin = float(self._sb.getExpertParameter('VLBIPriorMargin', default=0.))
        self._ForcePostCal = int(self._sb.getExpertParameter('ForcePostCal', default=0))
        self._SkipAlmaCal = int(self._sb.getExpertParameter('SkipAlmaCal', default=0))
        self._Regression = int(self._sb.getExpertParameter("Regression", default=0))
        self._LazyPointingQuery = int(self._sb.getExpertParameter("LazyPointingQuery", default=1))

        APP.APP_SetEPDefaults(self)
        APP.APP_SelfInit(self)

        # If True, then do not perform pointing scans for bandpass and polarization
        # targets. If antenna pointing models meet the specification (2" over whole
        # sky, if I remember correctly), then there should be no major problem for
        # not having pointing scans on bandpass and polarization targets.
        self._pointingOnce = int(self._sb.getExpertParameter("PointingOnce", default=0))
        self._maxPointingSeparation = float(
            self._sb.getExpertParameter('MaxPointingSeparation', default=45.))
        maxSep = "%f deg" % self._maxPointingSeparation
        from Observation.PointingCalTarget import PointingCalTarget
        PointingCalTarget.setMaximumSeparation(maxSep)
        self.logInfo("Max Pointing Sep was updated to " + maxSep)

        # For last-minute change of expert parameters.
        self._LastMinuteChangeFile = self._sb.getExpertParameter(
            "LastMinuteChangeFile", default="").strip()

        # capture initial expert parameter values
        self.ano(self, 'e', 'initial expert parameter values')

        self.logInfo("LastMinuteChangeFile = '%s'"%self._LastMinuteChangeFile)
        if self._LastMinuteChangeFile != "":
            rv = APP.APP_UpdateExpertParameters(
                self, self._LastMinuteChangeFile)
            self.logInfo('LastMin returned ' + str(rv))

        # Tweak some parameters for a regression test.
        if self._Regression == 1:
            import Observation.APPRegression as APPRegression
            # 'VLBISchedule', 'VLBISources', and 'VLBIDone' will be updated.
            self.logInfo("Updating schedule for start in %d min" % self._RegSchedMins)
            APPRegression.APP_UpdateParameters(self)
        else:
            self.logInfo("This is not a regression test")

        # check observing groups
        groupList = self._sb.getGroupList()
        if len(groupList) < 2:
            msg = "VLBI Scheduling Block requires two observing groups, " + \
                  "but only %d configured." % len(groupList)
            raise Exception(msg)

        # ICT-9846 -- this is probably always to be true but let us regularize
        # first and then clean out the crap on the path to Cycle9
        if self._LazyPointingQuery == 1:
            self.logInfo("Enable lazy evaluation of pointing query")
            for group in groupList:
                group.enableLazyPointingQuery(True)
        else:
            self.logInfo("Not being Lazy on pointing")

        APP.APP_SBConfig(self)
        APP.APP_SelfSetup(self)

    def prepare(self):
        '''
        This method creates the VOM, begins execution and performs
        any required initial calibration or other activities.
        '''
        self._prepTime = APP.getAcsTimestamp()
        self.ano(self, 'v', 'prepare ' + str(self._prepTime))
        self.logInfo('Observation Preparations')
        # pre-execution setup and checks
        self.array = self._array
        self.logInfo("Array is %s, _arrayName is %s" % (
            self.array, self._arrayName))
        APP.APP_TelCalParameters(self)
        APP.APP_AntennaConfig(self)
        APP.APP_ApsModeConfig(self)
        # Configure spectral spec
        APP.APP_SpectralCheck(self)
        # Make sure schedule and targets match
        APP.APP_TargetCheck(self)
        # Configure _VLBIScanList
        APP.APP_ScheduleCheck(self)
        APP.APP_ApsModeReport(self)
        APP.APP_SpecLineSetup(self)
        # Report on PIC and other VLBI readiness
        APP.APP_PICStatus(self)
        APP.APP_FinalRunTimeChecks(self)
        # Initialize the APS delay system from TMCDB as specified in the SB.
        self.programApsDelaySystem()
        #
        # After this point, we are messing with the array, so
        # exceptions may require some cleanup (e.g. destroy the array).
        self.array.setAppSumAntenna(True)
        self.array.beginExecution()
        self.ano(self, 'v', 'begin ' + str(APP.getAcsTimestamp()))
        # post-execution configuration
        self._ASDM = self._array.getExecBlockUID()
        self.logInfo("Beginning Execution, uid is %s" % (self._ASDM))
        self.reportShiftLogIfRequired()
        self.vom = self.array.getVLBIObservingMode()
        APP.APP_ShowHWSetup(self)
        # Configure maximum time for single SB execution
        endTime = self._sb.getMaximumExecutionTime()
        self.vom.setMaximumSchedBlockTime(endTime)
        # Add some SSR "ugliness"
        self.addSSRModetoVOM()
        # SSR: Identify VLBI targets that needs to be
        # assigned with OBSERVE_TARGET intent.
        self.identifyScienceScans()
        # SSR: If spectral specs for ALMA calibrations do
        # not match with VLBI specs, replace them.
        self.checkCalibrationSpectralSpecs()
        # SSR: Resolve calibrator queries
        self.expandSourceQueries()
        # SSR: Add CALIBRATE_FLUX intent to bandpass targets,
        # if amp target does not exist.
        self.tweakFluxCalIntent()
        # SSR: Add generated VLBI targets (_VLBIScanList) to group2
        self.addVLBITargetsToGroup2()
        # SSR: Create ATM cal targets
        self.createAtmCalTargets()
        # SSR: Select pointing sources
        self.selectPointingTargets()
        # SSR: Ugly sessions...
        if self._sessionControl:
            self.where = 'recycleAttenSettings()'
            self.vom.sbe.recycleAttenSettings()
            self._sb.setTargetsId()
            self.where = 'recycleFieldSources()'
            self.vom.sbe.recycleFieldSources()
        # Start interacting with TelCal
        APP.APP_TelCalConfig(self)
        # This is only incompletely handled in simulation
        try:
            self.updateTelCalForApsDelaySystem()
        except Exception as ex:
            self.logException('Problem Updating TelCal for Delays', ex)
        APP.APP_VOMConfig(self)
        self.ano(self, 'v', 'prepdone ' + str(APP.getAcsTimestamp()))

    def programApsDelaySystem(self):
        # assign a simulated cost to initialization
        self.where = 'programApsDelaySystem()'
        if 'OSS' in self._arrayName: self._array.incrementElapsedTime(15.0)
        self.logInfo("Initializing APS Delay system: how='%s' req=%d" %
                (self._AppDelayInit, self._AppDelayRequired))
        try:    # gather the up-front configuration information
            if self._AppDelayInit:
                self.logInfo("Using how=" + self._AppDelayInit)
                self.apsdelays = APS_Delays(how=self._AppDelayInit,
                    array=self._arrayName, tcp=self.tcparams)
            else:
                self.logInfo("Using online default")
                self.apsdelays = APS_Delays(array=self._arrayName,
                    tcp=self.tcparams)
        except Exception as ex: # if it fails, we generally soldier on
            self.logException("which provoked an exception:", ex)
            if self._AppDelayRequired == 1:
                self.logInfo('Unable to init REQUIRED APS Delay system')
                raise Exception("APS Delay Programming Fault.")
            else:
                self.logInfo('Using Neutered APS Delay system')
                self.apsdelays = APS_Delays(how='neutered', array=None)
        self.logInfo('APS Delay Status: ' + self.apsdelays.status);

    def updateAveragingModes(self, averagingModes):
        # TODO: note that ams[index] might be ONE:TWO:PER in the future
        where = self.where
        self.where = 'updateAveragingModes()'
        self.logInfo("updateAveragingModes(%s)" % averagingModes)
        ams = averagingModes.split(',')
        for index,bb in enumerate(range(1,5)):
            try:
                if   ams[index] == 'ONE': mode = 'ONE_PER_ANT'
                elif ams[index] == 'TWO': mode = 'TWO_PER_ANT'
                else:                     mode = 'PER_AVERAGE'
            except Exception as ex:
                mode = 'PER_AVERAGE'
                self.logException('ams is ' + str(ams), ex)
            parm = 'averagingMode_BB_%d'%bb
            self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                parm, mode, self.apsdelays.array, 1)
            self.logInfo("Updated %s to %s in TCP" % (parm, mode))
        self.where = where

    def updateTelCalForApsDelaySystem(self):
        self.logInfo('updateTelCalForApsDelaySystem initialization')
        # provide delay info and how to use it via TelCalParameters (tcp)
        self.where = 'updateTelCalForApsDelaySystem()'
        # self.applyBBDs acts as a global enable/disable
        self.applyBBDs = self._ApplyBBDs.split(',')
        # make sure we have all BBs populated with something
        for bb in range(len(self.applyBBDs),len(self._basebandEnums)):
            self.applyBBDs.append('off')
        # now proceed with adjusting parameter tunings
        # note that in simulation, this may die if TelCal isn't simulated
        self.updateAveragingModes(self._AveragingModes)
        self.apsdelays.tcp.parameterTuning.setCalibParameter(
            'applyPhasingSleepSecs', self._PhasingSleep,
            self.apsdelays.array, 1)
        ### FIXME delete the following line
        self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
            'averagingMode', self._AveragingMode, self.apsdelays.array, 1)
        # revise the IfSwitchMap from the programmed defaults.
        if self._IfSwitchMap != '':
            for bnd in self._IfSwitchMap.split(';'):
                band,hilolist = bnd.split(':')
                self.apsdelays.setIfSwitch(band, hilolist)
        # and only bother pushing delays if they are enabled on some BB
        if 'yes' in self.applyBBDs:
            haveRx = False
            self.logInfo("Using ALMA_RB list %s" % str(self.rbands))
            for rb in self.rbands:
                self.logInfo('Pushing Delay Values for Rx ' + rb)
                self.apsdelays.setDelayValuesBand(rb)
                haveRx = True
            if haveRx:
                if 'OSS' in self._arrayName:
                    self._array.incrementElapsedTime(120.0 * len(self.rbands))
                self.apsdelays.postDelayValues()
                msg = "Posted %d Delays to TelCal" % (self.apsdelays.postcount)
                self.logInfo(msg)
            else:
                self.logInfo('No Delay Values passed, disabling applyBBDs')
                self.applyBBDs = ['off','off','off','off']
        else:
            self.logInfo("Delay Fix was completely disabled")
        self.logInfo('Delay Fix: applyBBDs is ' + str(self.applyBBDs))
        self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
            'appDebugMode', self._TelCalDebug, self.apsdelays.array, 1)
        if self._TelCalDebug == '0': self.logInfo('No TelCal APP Debugging')
        else:                        self.logInfo('TelCal APP Debugging ON')
        #
        ## things we might teach TelCal to do in a later cycle
        #
        self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
            'delayGuidance', self._AppDelayGuidance, self.apsdelays.array, 1)
        self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
            'allowedPackModes', self._AllowedPackModes, self.apsdelays.array,1)
        self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
            'averagePols', self._AveragePols, self.apsdelays.array, 1)

    def updateApsCalibs(self, rx, vt):
        from Observation.APSDelayCalTarget import APSDelayCalTarget
        if self._AppDelayRequired != 1:
            self.logInfo('The APS Delay system is not required to be active')
            return
        if self._sessionControl:
            if 'APSDelayCalDone' in self.vom.sbe.exStateDict:
                self.logInfo('The Session claims this was already done')
                return
        if rx in self.apscalibs:
            self.logInfo('Already have a target for band ' + rx)
            return
        try:    # to initialize with EP parameter if supplied
            self.logInfo("using '" + str(self._ApsDelayCalParams) + "'")
            reps,dly = self._ApsDelayCalParams.split(',')
            irep = int(reps)
            fdly = float(dly)
            self.logInfo("using %d reps and %f delay" % (irep, fdly))
            thisRxCal = APSDelayCalTarget(
                rx=rx, vt=vt, repeats=irep, delay=fdly)
        except:
            self.logInfo('no usable APSDelayCalTarget params')
            thisRxCal = APSDelayCalTarget(rx=rx, vt=vt)
        if thisRxCal.isUsable:
            self.apscalibs[rx] = thisRxCal
            self.logInfo("Added %s (%s) (%s)" % (
                thisRxCal.personal, rx, str(list(self.apscalibs.keys()))))
        else:
            raise Exception('Unable to create APSDelayCalTarget')

    def identifyScienceScans(self):
        self.where = 'identifyScienceScans()'
        self.logInfo('Entering identifyScienceScans()')
        #self.rbands = set()
        group = self.getObservingGroup(2)
        for vlbiTarget in self._VLBIScanList:
            vlbiSS = vlbiTarget.getSpectralSpec()
            thisRxBand = str(vlbiSS.FrequencySetup.receiverBand)[-2:]
            self.rbands.add(thisRxBand)
            for target in group.getScienceTargetList():
                if (target.getSpectralSpec() == vlbiSS and
                    target.getSource() == vlbiTarget.getSource()):
                    self.logInfo(("Mark '%s' as science source " +
                        "(matched with '%s')") % (vlbiTarget, target))
                    vlbiTarget.isScienceSource = True
                    self.updateApsCalibs(thisRxBand, vlbiTarget)
                    break
                elif (self._Regression == 1 and
                    target.getSpectralSpec() == vlbiSS and
                    target.getSource().sourceName == 
                    vlbiTarget.getSource().sourceName):
                    self.logInfo(("Mark '%s' as science source (Regression) " +
                        "(matched with '%s')") % (vlbiTarget, target))
                    vlbiTarget.isScienceSource = True
                    self.updateApsCalibs(thisRxBand, vlbiTarget)
                    break
                else:
                    self.logInfo("%s == %s and %s == %s" % (
                        str(target.getSpectralSpec()), str(vlbiSS),
                        str(target.getSource()), str(vlbiTarget.getSource())))
                    self.logInfo("VLBI '%s' is not Science '%s'" % (
                        vlbiTarget, target))
        if len([vT for vT in self._VLBIScanList if vT.isScienceSource]) > 0:
            return
        self.logInfo("None of VLBI targets matched with science targets.")

    def tweakFluxCalIntent(self):
        groupList = self._sb.getGroupList()
        if sum([len(group.getAmplitudeCalTargetList()) for group in groupList]) > 0:
            return
        self.logInfo("No amplitude target defined: Will CALIBRATE_FLUX intent to bandpass target")
        for group in groupList:
            for target in group.getBandpassCalTargetList():
                target.setFluxCalReduction(True)

    def reportShiftLogIfRequired(self):
        self.logInfo("_scriptName=%s" % self._scriptName)
        self.logInfo("_scriptArgs=%s" % self._scriptArgs)
        try:
            self.logInfo("Shiftlog ASDM %s script %s args %s..." %
                (self._ASDM, self._scriptName, self._scriptArgs))
            self.shiftlog(self._ASDM, self._scriptName,
                scriptArgs=self._scriptArgs)
        except Exception as ex:
            self.logException("No Shiftlog on: ASDM %s script %s args %s..." %
                (self._ASDM, self._scriptName, self._scriptArgs), ex)

    def addSSRModetoVOM(self):
        '''
        An ugly hack shared among SSR observation scripts.
        '''
        self.where = 'addSSRModetoVOM()'
        import Observation.SBExecState
        from Observation.SBExecutionMode import SBExecutionMode
        from Observation.Global import addClassToMode
        addClassToMode(self.vom, SBExecutionMode)
        sbe = Observation.SBExecState.SBExecState(self._array, self._sb)
        self.vom.addSBExecState(sbe)
        if self._sessionControl:
            self.vom.sbe.startSession()

    def checkCalibrationSpectralSpecs(self):
        vlbiSpecs = [vlbiTarget.getSpectralSpec() for vlbiTarget in self._VLBIScanList]
        vlbiSpecs = list(set(vlbiSpecs))
        if len(vlbiSpecs) > 1:
            #Relaxing slightly...
            #raise Exception('Multiple spectral specs for ' +
            #   'VLBI targets are not supported.')
            vlbiSpecs = self._VLBISpecSpec
            self.logInfo('Multiple spectral specs are generally not '
                'supported on VLBI targets; but we shall soldier on with the '
                'VEX2VOM designated spectral spec' + vlbiSpecs[0].entityPartId)
        if len(vlbiSpecs) < 1 and self._VLBIOptional == 1:
            self.logWarning("No VLBI Targets; calibrations may be incorrect")
            return

        targets = []
        for group in self._sb.getGroupList():
            targets.extend(group.getAmplitudeCalTargetList())
            targets.extend(group.getBandpassCalTargetList())
            targets.extend(group.getPhaseCalTargetList())
            targets.extend(group.getPolarizationCalTargetList())
            #targets.extend(group.getCheckSourceCalTargetList())
            targets.extend(group.getDelayCalTargetList())

        vlbiSS = vlbiSpecs[0]
        for target in targets:
            ss = target.getSpectralSpec()
            if ss not in vlbiSpecs:
                self.logWarning("Spectral spec for '%s' does not match for any of VLBI spec(s)" % target)
                target.setSpectralSpec(vlbiSS)

    def getObservingGroup(self, groupNumber):
        groups = self._sb.getGroupList()
        return groups[groupNumber - 1]

    def addVLBITargetsToGroup2(self):
        group = self.getObservingGroup(2)
        for vlbiTarget in self._VLBIScanList:
            group.addTarget(vlbiTarget)

    def createAtmCalTargets(self):

        def register(targetCycleTimeDict, target, cycleTime=None):
            """
            Helper function for adding ATM cal to the target
            """
            if target.hasQuery():
                # Queries should have already resolved at this stage.
                return
            tssp = target.getSpectralSpec()
            band = self._sb.getBand(target)
            if cycleTime is None:
                if band in [1, 2, 3, 4, 5, 6]:
                    cycleTime = 600.0
                elif band == 7:
                    cycleTime = 480.0
                elif band == 8:
                    cycleTime = 420.0
                elif band in [9, 10]:
                    cycleTime = 360.0
                else:
                    raise Exception("Invalid band number %d [%s]" % (band))
            if target in targetCycleTimeDict:
                self.logInfo('[createAtmCalTargets] [%-60s] cycleTime=%6.1f oldCycleTime=%6.1f' %
                            (target, cycleTime, targetCycleTimeDict[target]))
            else:
                self.logInfo('[createAtmCalTargets] [%-60s] cycleTime=%6.1f' % (target, cycleTime))
            targetCycleTimeDict[target] = cycleTime

        self.logInfo("[createAtmCalTargets] Create ATM targets for ALMA calibrations scans.")

        atmSSCache = dict()
        fsToAtmTargetDict = dict()
        groupList = self._sb.getGroupList()
        for group in groupList:
            targetCycleTimeDict = dict()
            # Do not add ATM to phase calibrator or CheckSourceTargets, &c
            # but do make sure ATM ends up on VLBI, and Pol, Amp & BP.
            if group.groupIndex == 2:
                [register(targetCycleTimeDict, target) for target in self._VLBIScanList]
            [register(targetCycleTimeDict, target) for target in group.getPolarizationCalTargetList()]
            [register(targetCycleTimeDict, target) for target in group.getAmplitudeCalTargetList()]
            [register(targetCycleTimeDict, target) for target in group.getBandpassCalTargetList()]
            # If 'forceATM' is checked on OT, set cycle time to 6 seconds.
            [register(targetCycleTimeDict, target, cycleTime=6.) for target in group.getTargetList() if target.forceAtm]
            # Construct ATM cal target instances
            for target, cycleTime in targetCycleTimeDict.items():
                isVLBITarget = target in self._VLBIScanList
                fs = target.getSource()

                # Share same ATM target instance among VLBI scans with a same source, so that
                # ATM scans will follow cycle time rule.
                shareATM = isVLBITarget and (fs in fsToAtmTargetDict)
                if shareATM:
                    atmTarget = fsToAtmTargetDict[fs]
                else:
                    atmTarget = group.constructATMTarget(target, cycleTime,
                                                         atmSpectralSpecCacheDict=atmSSCache)
                    if atmTarget is None:
                        continue
                    # ICT-7734: tweak ATM reference position
                    atmTarget.tweakReferenceOffset(verbose=False, saveOffsetToFS=True)
                    # set true in case the fast mode is to be used
                    atmTarget.setApplyWVR(True)
                    # and reduce the timeout to something tolerable
                    atmTarget.wvrResultTimeout = 8  # seconds
                # once is enough
                if not atmTarget in target.getAssociatedCalTarget():
                    target.addAssociatedCalTarget(atmTarget)
                if isVLBITarget:
                    fsToAtmTargetDict[fs] = atmTarget

            group.printTargets("Added ATM to normal targets")

        # Print target list
        for group in groupList:
            group.printTargets("AFTER createAtmCalTargets")

    def expandSourceQueries(self):
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing', useSSRLogger=True)
        ### for Cy6 only
        try:     cc.setObsmode(self.vom)
        except:  self.logInfo('[regression] unable to set VOM in CalibratorCatalog')
        ###
        for group in self._sb.getGroupList():
            # Perform doppler correction for spectral specs
            self._sb.adjustDopplerByGroup(self.vom, group)
            # Resolve target queries
            if self._Regression == 1:
                self.logInfo("[regression] removing polarization queries for safety")
                for target in group.getPolarizationCalTargetList():
                    self.logInfo("[regression] removing %s" % str(target))
                    group.removeTarget(target)
            else:
                self.logInfo("processing normally...")
            cc.resolveGroupQueries(group)

    def selectPointingTargets(self):
        groupList = self._sb.getGroupList()

        # Identify observing group that has pointing target.
        # (In the future, we have to make a change request
        # to OT team so that pointing target will be populated
        # under both group1 and group2.)
        groupWithPnt = None
        for group in groupList:
            hasPointing = len(group.getPointingCalTargetList()) > 0
            self.logInfo("Group%d hasPointing=%s" % (group.groupIndex, hasPointing))
            if hasPointing:
                groupWithPnt = group
                break
        else:
            self.logInfo("No pointing target defined: will not perform pointing scans")
            return

        # Cache dictionary: Key is FieldSource and value is PointingCalTarget
        fsToPntTargetDict = dict()
        # Identify which targets to add pointing target
        targets = []
        if self._pointingOnce:
            # Just add pointing target to VLBI targets only.
            targets.extend(self._VLBIScanList)
        else:
            # Add pointing to every targets.
            for group in groupList:
                targets.extend(group.getPhaseCalTargetList())
                targets.extend(group.getPolarizationCalTargetList())
                targets.extend(group.getAmplitudeCalTargetList())
                targets.extend(group.getBandpassCalTargetList())
                #targets.extend(group.getCheckSourceCalTargetList())
                targets.extend(group.getDelayCalTargetList())
            targets.extend(self._VLBIScanList)

        for target in targets:
            pntTarget = groupWithPnt.selectPointingCalTarget(target, cacheDict=fsToPntTargetDict)
            if pntTarget is None:
                self.logInfo("[selectPointingTargets] no pointing source for '%s'" % (target))
                continue
            self.logInfo("[selectPointingTargets] selected %s for '%s'" % (pntTarget.getSource().sourceName, target))
            pntTarget.setUseReferencePosition(False)
            target.addAssociatedCalTarget(pntTarget, 0)
            fsToPntTargetDict[target.getSource()] = pntTarget
            # Disable TelCal results application on SCO simulated STEs...
            if os.environ["LOCATION"].startswith("SCO"):
                self.logInfo("LOCATION='%s' : disable results application" % (os.environ["LOCATION"]))
                pntTarget.setApply(False)
            # Disable automatic adjustment of pointing subscan duration
            pntTarget.enableSubscanDurationOptimization(False)

    def getEntryMargin(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return (vlbiTarget.CorrStartTime - curTime) / 10000000.0

    def getTimeAndMargin(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return curTime, (vlbiTarget.CorrStartTime - curTime) / 10000000.0

    def getTimeAndMarginExit(self, vlbiTarget):
        curTime = APP.getAcsTimestamp()
        return curTime, (vlbiTarget.CorrEndTime - curTime) / 10000000.0

    def estimateDuration(self, target):
        interSubscanLatency = 1.5
        # TODO: update and these values, if required
        # apply pointing results => 8 seconds?
        # attenuatorOptimization = 0.
        # correlatorCalibration = 0.
        # tSetPointingDirection = 4.
        # tCorrCalibration = 4.
        # tTune = 23.
        # tInterSubscan = 1.5
        # tTelCal = 8

        target._populateSubscanList()
        subscanSpec = target.getSubscanSequence()
        subscanDurations = [subscan.duration for subscan in subscanSpec.subscans]

        duration = sum(subscanDurations)
        duration += len(subscanDurations) * interSubscanLatency

        # Add associated calibration targets
        for aTarget in target.getAssociatedCalTarget():
            isNeeded = aTarget.isNeeded(self.vom)
            if not isNeeded:
                continue
            className = aTarget.__class__.__name__
            if className == 'AtmCalTarget':
                duration += 90.
            elif className == 'PointingCalTarget':
                duration += 210.
            else:
                self.logWarning("[estimateDuration] unknown calibration type '%s'" % (aTarget))
        return duration

    def targetTimingCheck(self):
        t1 = self.vom.sbe.getElapsedTime()
        t2 = self.vom.getElapsedTime()
        self.logInfo("[targetTimingCheck] sbeET() %.1f vomET() %.1f"%(t1, t2))

    def checkCalibrationTargets(self, group):
        targets = []
        targets.extend(group.getPhaseCalTargetList())
        targets.extend(group.getPolarizationCalTargetList())
        targets.extend(group.getAmplitudeCalTargetList())
        targets.extend(group.getBandpassCalTargetList())
        # targets.extend(group.getCheckSourceCalTargetList())
        targets.extend(group.getDelayCalTargetList())
        targetAndDurationList = []
        if 'OSS' in self._arrayName: self.targetTimingCheck()
        for target in targets:
            isObservable = target.isObservable(self.vom, duration=600)
            isNeeded = target.isNeeded(self.vom)
            if not isObservable or not isNeeded:
                continue
            duration = self.estimateDuration(target)
            self.logInfo("[checkCalibrationTargets] estimated duration=%5.1f sec '%s' integration time=%5.1f sec" % (duration, target, target.getIntegrationTime()))
            targetAndDurationList.append((target, duration))
        return targetAndDurationList

    def doPointingIfRequired(self, target):
        '''
        This method does pointing for the target (if required)
        '''
        pntTargets = []
        for aTarget in target.getAssociatedCalTarget():
            if aTarget.__class__.__name__ == 'PointingCalTarget':
                pntTargets.append(aTarget)
        # There should only 1 (or 0) pointing target associated.
        if len(pntTargets) == 0:
            return
        pntTarget = pntTargets[0]
        if not pntTarget.isNeeded(self.vom):
            return

        # disable phasing and restore standard delay handling
        self.logInfo('AppMode type "calalma"')
        self.vom.setAppScanParameters(
            self._appModeSeq['calalma'], self._PackMode)

        self.logInfo("Perform pointing ... '%s'" % (pntTarget))
        pntTime, entry = self.getTimeAndMargin(target)
        pntTarget.execute(self.vom)
        self.apsisflat = [False for x in range(0,4)]
        curTime, exit = self.getTimeAndMargin(target)
        self.describeCal('<point>', pntTarget, pntTime, curTime, entry, exit)

    def executeCalTarget(self, target, action="<calib>", force=False):
        '''
        Execute ALMA specific calibration scan. Rather than just calling
        target.execute(), this method used executeAssociatedCalTargetList()
        so that timing information targets can be recorded.
        '''
        isObservable = target.isObservable(self.vom, duration=600)
        isNeeded = target.isNeeded(self.vom) or force
        if not isObservable or not isNeeded:
            self.logWarning("isObservable=%d isNeeded=%d skip '%s'" % (isObservable, isNeeded, target))
            return
        # Execute associated calibration targets (pointing or ATM).
        self.executeAssociatedCalTargetList(target)
        # Temporily evacuate associated calibration targets (just in case).
        assocTargetsBackup = list(target.getAssociatedCalTarget())
        target.clearAssociatedCalTarget()

        # disable phasing and restore standard delay handling
        self.logInfo('AppMode type "calalma"')
        self.vom.setAppScanParameters(
            self._appModeSeq['calalma'], self._PackMode)

        # Execute the calibration
        eTime, entry = self.marginHelper(self._FirstVLBITarget)
        target.execute(self.vom)
        self.apsisflat = [False for x in range(0,4)]
        cTime, exit = self.marginHelper(self._FirstVLBITarget)
        self.describeCal(action, target, eTime, cTime, entry, exit)
        # Restore associated calibration targets
        [target.addAssociatedCalTarget(aT) for aT in assocTargetsBackup]

    def marginHelper(self, target):
        from Observation.VLBICalTarget import VLBICalTarget
        if isinstance(target, VLBICalTarget):
            time, margin = self.getTimeAndMargin(target)
        elif self._FirstVLBITarget:
            time, margin = self.getTimeAndMargin(self._FirstVLBITarget)
        else:
            time, margin = APP.getAcsTimestamp(), 0.
        return time, margin

    def executeAssociatedCalTargetList(self, target, action="<assoc>"):
        for aTarget in target.getAssociatedCalTarget():
            className = aTarget.__class__.__name__
            isNeeded = aTarget.isNeeded(self.vom)
            self.logInfo("[assoc] isNeeded=%s '%s'" % (isNeeded, aTarget))
            if not isNeeded:
                continue
            if className == "PointingCalTarget":
                if self._SkipAlmaCal == 1:
                    self.logInfo("[assoc] skipping pointing as directed")
                    continue
                APP.APP_TestingPointing(self)
            elif className == "AtmCalTarget":
                APP.APP_TestingAtm(self)
            else:
                self.logWarning("Unsupported target for associated calibrations: '%s'" % (aTarget))

            # disable phasing and restore standard delay handling
            self.logInfo('AppMode type "calalma"')
            self.vom.setAppScanParameters(
                self._appModeSeq['calalma'], self._PackMode)

            eTime, entry = self.marginHelper(target)
            aTarget.execute(self.vom)
            self.apsisflat = [False for x in range(0,4)]
            cTime, exit = self.marginHelper(target)
            self.describeCal(action, aTarget, eTime, cTime, entry, exit)

    def calibratePriorToVLBILoop(self):
        """
        Execute group1 targets (should be ALMA specific calibrations)
        """
        self._FirstVLBITarget = None
        self.where = 'calibratePriorToVLBILoop()'
        if self._pointingOnce:
            # If _pointingOnce=True, then Perform pointing scan only for
            # the first VLBI source.
            for vlbiTarget in self._VLBIScanList:
                if self.getEntryMargin(vlbiTarget) < self._VLBICalMargin:
                    continue
                self._FirstVLBITarget = vlbiTarget
                APP.APP_TestingPointing(self)
                self.doPointingIfRequired(vlbiTarget)
                break
        elif len(self._VLBIScanList) > 0:
            # For margin calculations within the following, we define:
            self._FirstVLBITarget = self._VLBIScanList[0]

        # Perform as many as possible group1 targets before the first
        # VLBI scan, if any
        self.calibratePriorToScan(self._FirstVLBITarget, groupIndex=1,
                force=self._ForceGroup1Calibration)

    def contemplateGroupOneTargets(self, margin):
        '''
        This is will consider trying harder on group 1 targets
        (Amplitude and Bandpass) if ForceGroup1Interleave was set.
        The option remains to be implemented what to do if this
        parameter is > 1.  This allows increasingly more agressive
        steps to be taken.
        '''
        self.logInfo('[contemplateGroupOneTargets] enable is %d' %
            self._ForceGroup1Interleave)
        if self._ForceGroup1Interleave == 0: return None
        group = self.getObservingGroup(1)
        # if self._ForceGroup1Interleave > 1: be more agressive
        for target, duration in self.checkCalibrationTargets(group):
            if duration > margin:
                continue
            self.logInfo('[contemplateGroupOneTargets] got one')
            # FIXME: if the next VLBI target is not the science target
            # then consider starting late on the VLBI scan to get this cal.
            return target
        return None

    def calibratePriorToScan(self, vlbiTarget, groupIndex=2, force=0):
        '''
        This method is called to arrange calibrations prior to the next (VLBI)
        scan.  The most important thing is to be done (with margin) before
        the correlator must be running the scan.  The first scan is special
        in that various one-time calibrations happen before VLBI begins.
        The VLBI scan is presented to provide information about the scan,
        and most importantly, when it is scheduled to start.
        '''
        # note next target for diagnostics
        self.where = 'calibratePriorToScan()'
        self._FirstVLBITarget = vlbiTarget

        group = self.getObservingGroup(groupIndex)
        # If there is a margin, perform some ALMA calibration scans.
        # self._VLBICalMargin may be set large to give priority to cals
        while self._SkipAlmaCal == 0:
            if force == 1:
                margin = self._maxSchedMargin
                self.logInfo('[calibratePriorToScan] Forcing calibrations')
            elif vlbiTarget:
                margin = self.getEntryMargin(vlbiTarget)
                self.logInfo('[calibratePriorToScan] Waiting for %s at %s '
                    '[margin=%.1f sec < %.1f ?]' % (vlbiTarget.SourceName,
                    vlbiTarget.RecVexTime, margin, self._VLBICalMargin))
                if margin < self._VLBICalMargin:
                    break
            else:
                margin = self._maxSchedMargin
                self.logInfo('[calibratePriorToScan] No VLBI Scan is pending')

            # Pick up an appropriate target to execute
            targetToExecute = None
            for target, duration in self.checkCalibrationTargets(group):
                if duration > margin:
                    continue
                targetToExecute = target

            # A hook to allow group one targets in this case
            if targetToExecute is None and groupIndex == 2:
                targetToExecute = self.contemplateGroupOneTargets(margin)

            # we really have nothing to do here
            if targetToExecute is None:
                self.logInfo('[calibratePriorToScan] Has a margin of '
                    + '%.1f sec, but there is nothing else to slip in' %
                        (margin))
                break

            self.logInfo('[calibratePriorToScan] picked up "%s"' %
                targetToExecute)
            self.executeCalTarget(targetToExecute, action='<calib>')

        if groupIndex == 1:
            return

        # if there are no VLBI scans, we should not be here, but j.i.c:
        if vlbiTarget is None:
            return

        # Just for in case, check exit margin for the next VLBI scan.
        # If it is too late, it does not make sense to perform calibrations.
        curTime, exitMargin = self.getTimeAndMarginExit(vlbiTarget)
        self.logInfo("[calibratePriorToScan] exit margin for next VLBI scan: %.1f [seconds]" % (exitMargin))
        #if exitMargin < 0.:
        if exitMargin < self._VLBIPriorMargin:
            self.logInfo("[calibratePriorToScan] It is too late and thus will not perform any calibration for '%s'" % vlbiTarget)
            return

        try:
            from Observation.CalibratorSource import getTimeToRise
            import time
            src = vlbiTarget.getSource()

            # self._maxSchedMargin is how long we were prepared to
            # wait for VLBI scans so it's a similar sort of limit
            if self._TimeToRise > self._maxSchedMargin:
                msg = "reducing TimeToRise %.1f -> %.1f" % (
                    self._TimeToRise, self._maxSchedMargin)
                self.logInfo("[calibratePriorToScan] " + msg)
                self._TimeToRise = self._maxSchedMargin

            timeToRise_ = getTimeToRise(src, elLimit=self._ElLimit)
            msg = "Time to rise = %.1f [seconds] ('%s')" % (
                timeToRise_, vlbiTarget)
            self.logInfo("[calibratePriorToScan] %s" % msg)
            # if timeToRise_ is not a huge positive number, wait for it.
            while 0 < timeToRise_ and timeToRise_ < self._TimeToRise:
                msg = "sleep(10.0); 0 < %.1f < %.1f" % (
                    timeToRise_, self._TimeToRise)
                self.logInfo("[calibratePriorToScan] " + msg)
                time.sleep(10.0)
                timeToRise_ = getTimeToRise(src, elLimit=self._ElLimit)
        except:
            # Just for in case...
            import traceback
            self.logInfo("%s" % (traceback.format_exc()))

        # Then do associated calibrations for VLBI target (pointing or ATM).
        self.executeAssociatedCalTargetList(vlbiTarget)

    def calibrateAfterScans(self):
        '''
        If we are not yet at the appointed time, i.e. self._AcsDoneTime
        which is calculated from VLBIDone, and if there are calibrations
        that remain to be done, then do them now.  VLBIDone can be set
        into the future to force this calibration at the end.
        '''
        if self._SkipAlmaCal == 1:
            self.logInfo('skipping post-schedule calibrations')
            return
        #
        self.where = 'calibrateAfterScans()'
        self.logInfo('Expecting to Finish by VLBIDone: ' + self._ThisVLBIDone)
        curTime = APP.getAcsTimestamp()
        if curTime > self._AcsDoneTime:
            self.logInfo('No time left for residual calibrations')
            if self._ForcePostCal == 0:
                return
            else:
                self.logInfo('ForcePostCal asserted, proceeding.')
        #
        self.logInfo('Will perform final polarization calibration')
        group = self.getObservingGroup(2)
        self._FirstVLBITarget = None
        for target in group.getPolarizationCalTargetList():
            self.executeCalTarget(target, action="<after>", force=True)

    def describeCal(self, action, target, atime, curTime, entry, exit):
        '''
        Enquiring minds will want to know how we did
        '''
        # if action == '':
        #     className = target.__class__.__name__
        #     action = className.replace("CalTarget", "").lower()
        #     action = "<%s>" % (action.replace("ing", ""))
        dwell = int(float(curTime - atime + 0.5) / 10000000.0)
        outcome = '%s (%ds)' % (target, dwell)
        self._obsTimes.append((curTime, entry, exit, action, outcome))
        self.ano(self, 'c', "cali times %d %d %d %s %s" % (
            curTime, entry, exit, action, outcome))

    def describeScan(self, vlbiScan, curTime, dwell, entry, exit):
        '''
        Enquiring minds will want to know how we did.  In the arguments
        curTime is when we got back from execute, the variable 'did' below
        is how much time the correlator should have been busy.  The
        time system for curTime is from APP.getAcsTimestamp()
        '''
        if vlbiScan.ActivePhase:
            vmode = 'Active  ' + vlbiScan.ApsMode
        else:
            vmode = 'Passive ' + vlbiScan.ApsMode
        if exit == None:
            outcome = 'skip %s at %s (0s/0s/0.0s) (%s)' % (
                vlbiScan.SourceName, vlbiScan.RecVexTime, vmode)
            self.logInfo('%s [%.3f]' % (outcome, -entry))
            exit = 0.0
        else:
            APP.APP_ReportScanStatus(self)
            did = (vlbiScan.CorrEndTime - vlbiScan.CorrStartTime) / 10000000.0
            vexCStart = APP.vexAcsTimestamp(int(curTime - 10000000.0 * did))
            vexFinish = APP.vexAcsTimestamp(int(curTime))
            outcome = 'vlbi %s at %s (%ds/%ds/%.3fs) (%s)' % (
                vlbiScan.SourceName, vlbiScan.RecVexTime,
                dwell, int(vlbiScan.ScanSeconds), did, vmode)
            self.logInfo("# vlbi %s at %s (actual vex corr start)" % (
                vlbiScan.SourceName, vexCStart))
            # negative exit is good: it means we exited later than
            # the planned exit time and didn't cut the scan short
            self.logInfo('# %s [%.3f %.3f]' % (outcome, entry, -exit))
            self.logInfo("# vlbi %s at %s (actual vex corr finish)" % (
                vlbiScan.SourceName, vexFinish))
        self._obsTimes.append(  # ditto on exit
            (curTime, entry, -exit, vlbiScan.ScanName, outcome))
        self.ano(self, 'v', "vlbi times %d %d %d %s %s" % (
            curTime, entry, -exit, vlbiScan.ScanName, outcome))

    def performVlbiScan(self, vlbiScan, rx=''):
        '''
        This method is called to execute the provided VLBI scan.
        '''
        self.where = 'performVlbiScan()'
        self.logInfo('performing  %s at %s, AppMode type %s' % (
            vlbiScan.SourceName, vlbiScan.RecVexTime, vlbiScan.ApsMode))
        curTime, entryMargin = self.getTimeAndMargin(vlbiScan)
        dwell = curTime
        exitMargin = None
        if (self._lateMargin < 0):
            vlbiScan.reduceSubscanRepeats(
                self._entryMargin, entryMargin, self._lateMargin)
            entryMargin = self.getEntryMargin(vlbiScan)
        # VEX might consider it observable, but ALMA rules here.
        # APS Cal is by definition observable (already checked)
        isObservable = vlbiScan.isObservable(self.vom,
            duration=vlbiScan.ScanSeconds)
        if not isObservable:
            self.logWarning('NOT OBSERVABLE (entry margin %.3f)' % entryMargin)
            entryMargin = 0.0
        APP.APP_TestingScan(self, vlbiScan, 0)
        # Temporily evacuate associated calibration targets (pointing or ATM),
        # so that they will not intrude into a VLBI scan.
        assocTargetsBackup = list(vlbiScan.getAssociatedCalTarget())
        vlbiScan.clearAssociatedCalTarget()
        if entryMargin > self._entryMargin:
            self.preScanDelayWork(self._apsModes[vlbiScan.ApsMode], rx=rx)
            vlbiScan.setWVRReduction()
            vlbiScan.checkRates(self._numBasebands, self._numberAntennas)
            vlbiScan.specLineModify(self._SpecLineBB1Only)
            self.vom.setAppScanParameters(
                self._appModeSeq[vlbiScan.ApsMode], self._PackMode)
            scanList = Observation.ScanList.ScanList()
            for xx in range(vlbiScan.CorrSubscanRepeats):
                vlbiScan.execute(self.vom, scanList)
            APP.APP_TestingScan(self, vlbiScan, 1)
            scanList.execute(self.vom, startTime=vlbiScan.CorrStartTime)
            curTime, exitMargin = self.getTimeAndMarginExit(vlbiScan)
            dwell = int(float(curTime - dwell + 0.5) / 10000000.0)
            vlbiScan.specLineReturn(self._SpecLineBB1Only)
            vlbiScan.restoreRates()
            self.postScanDelayWork(self._apsModes[vlbiScan.ApsMode], rx=rx)
        self.describeScan(vlbiScan, curTime, dwell, entryMargin, exitMargin)
        # Restore associated calibration targets
        [vlbiScan.addAssociatedCalTarget(aT) for aT in assocTargetsBackup]

    def preCalScanDelayWork(self, apsPars, rx):
        '''
        Processing prior to the APS-Cal scan: set 'cal' and comment.
        '''
        self.logInfo('preCalScanDelayWork for rx ' + rx)
        for bbe in self._basebandEnums:
            try:
                self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                    "applyBBDelay_" + str(bbe), 'cal', self.apsdelays.array, 1)
                self.updateAveragingModes('PER,PER,PER,PER')
                ### FIXME: delete the following
                self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                    'averagingMode', 'PER_AVERAGE', self.apsdelays.array, 1)
            except Exception as ex:
                self.logException("Problem tuning for delay cal", ex)
        self.logInfo('preCalScanDelayWork completed for rx ' + rx)

    def postCalScanDelayWork(self, apsPars, rx):
        '''
        Processing after the APS-Cal scan: should be 'sane' or 'fail'
        self.delaycalscanpass is set to False on anything but 'sane'
        The timer is to give TelCal a chance to reply.
        Normal responses may take a few seconds.
        '''
        import time
        self.logInfo('postCalScanDelayWork for rx ' + rx)
        self.delaycalscanpass = True
        for bbe in self._basebandEnums:
            try:
              for ww in range(10):
                self.logInfo('Waiting for TelCal to digest...')
                time.sleep(0.50)
                rep = \
                self.apsdelays.tcp.parameterTuning.getCalibParameterAsString(
                    "applyBBDelay_" + str(bbe), self.apsdelays.array)
                if rep[0:3] != 'cal':
                    self.logInfo("... got '%s' on %s" % (rep,str(bbe)))
                    break
            except Exception as ex:
                rep = 'exception'
                self.logException('Problem getting delay cal status', ex)
                # no reason to raise an Exception, treat as '!sane'
            rep,info = APP.APP_TestingCalScanReport(self, rep)
            msg = 'TelCal lobbed a curve ball'
            if rep == 'cal':   msg = 'TelCal did not respond (cal)'
            if rep == 'sane':  msg = 'TelCal has a good solution (sane)'
            if rep == 'tmcdb': msg = 'TelCal will be using TMCDB (tmcdb)'
            if rep == 'fail':  msg = 'TelCal was not very happy (fail)'
            self.logInfo('CalScanResult: "%s(%s)" for %s:%s, %s' % (
                rep, info, rx, str(bbe), msg))
            if rep != 'sane': self.delaycalscanpass = False

    def preScanDelayWork(self, apsPars, rx=''):
        '''
        TelCal understands 'off' (Cycle4) 'yes' (fix it) 'no' (already fixed)
        tcFixBBDs controls whether the new logic is 'off' or 'on'

        If called with non-empty rx value (RB), then this is for 'cal' scan
        If phases are not to be kept, then the phase registers are reset.
        '''
        if rx != '':
            self.preCalScanDelayWork(apsPars, rx)
            return
        self.logInfo('apsisflat:' + str(self.apsisflat) + ' preScanDelay')
        self.logInfo(str(apsPars) + ' VOMEnabled: ' + str(apsPars.vomEnable))
        averagingModes = ''
        intended_Modes = self._AveragingModes.split(',')
        self.logInfo('intended averaging modes: ' + str(intended_Modes))
        for bbe in self._basebandEnums:
            bbi = apsPars.indexOf(bbe)
            bbd = self.applyBBDs[bbi]
            rep = "set:" + bbd
            # If the per-scan instruction is 'on' or 'yes',
            # then it doesn't matter if bbd is already 'off';
            # however, we still need to update the averaging mode
            if bbd == 'off':
                averagingModes += ',PER'
                #self.logInfo('averagingModes is ' + averagingModes)
            if bbd == 'yes' and apsPars.tcFixBBDs[bbi] == 'off':
                bbd = 'off'
                rep += "(tfix) -> " + bbd
                averagingModes += ',PER'
                #self.logInfo('averagingModes is ' + averagingModes)
            # disable delay correction if it is already flat:
            if bbd == 'yes' and self.apsisflat[bbi]:
                if apsPars.keptPhases[bbi]:
                    bbd = 'no'
                    rep += '(flat/kept) -> ' + bbd
                else:
                    bbd = 'yes'
                    rep += '(flat/zero) -> ' + bbd
                averagingModes += ',' + intended_Modes[bbi]
                #self.logInfo('averagingModes is ' + averagingModes)
            if bbd == 'yes' and not self.apsisflat[bbi]:
                bbd = 'yes'
                rep += '(slope/zero) -> ' + bbd
                averagingModes += ',' + intended_Modes[bbi]
                #self.logInfo('averagingModes is ' + averagingModes)
            # temporarily mark self.apsisflat[bbi] with a string
            self.apsisflat[bbi] = bbd
            #self.logInfo('averagingModes is ' + averagingModes)
            try:
                self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                    "applyBBDelay_" + str(bbe), bbd, self.apsdelays.array, 1)
                ### FIXME: delete the following line
                self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                    'averagingMode',self._AveragingMode,self.apsdelays.array,1)
                self.logInfo("Tuned: applyBBDelay_" + str(bbe) + ' ' + bbd)
                self.apsdelays.tcp.parameterTuning.setCalibParameterAsString(
                    "phasingReport_" + str(bbe), rep, self.apsdelays.array, 1)
                self.logInfo("(set-BBD)phasingReport_%s '%s'"%(str(bbe),rep))
            except Exception as ex:
                self.logException("Problem tuning for delays", ex)
        self.logInfo('averagingModes is ' + averagingModes)
        self.updateAveragingModes(averagingModes[1:])
        self.logInfo('apsisflat:' + str(self.apsisflat) + ' preScanDelay')

    def postScanDelayWork(self, apsPars, rx=''):
        '''
        After scans, get a report from TelCal on how things stand.
        '''
        if rx != '':
            self.postCalScanDelayWork(apsPars, rx)
            return
        self.logInfo("apsisflat:" + str(self.apsisflat) + ' postScanDelay')
        self.bbdreps = list(range(4))
        for bbe in self._basebandEnums:
            bbi = apsPars.indexOf(bbe)
            try:
                rep = \
                self.apsdelays.tcp.parameterTuning.getCalibParameterAsString(
                    "phasingReport_" + str(bbe), self.apsdelays.array)
                self.logInfo("(get-BBD)phasingReport_%s '%s'"%(str(bbe),rep))
            except Exception as ex:
                rep = 'nada:no status'
                self.logException("Problem getting delay status", ex)
            self.bbdreps[bbi] = rep
            # update flatness state -- only correct if we have a valid report
            # in the off case, it's less confusing to declare it flat
            # True may turn to false below
            if   self.apsisflat[bbi] == 'yes': self.apsisflat[bbi] = True
            elif self.apsisflat[bbi] == 'no':  self.apsisflat[bbi] = True
            elif self.apsisflat[bbi] == 'off': self.apsisflat[bbi] = True
            else:                              self.apsisflat[bbi] = False
        self.logInfo("apsisflat:" + str(self.apsisflat) + ' postScanDelay')
        self.delayPanic(apsPars)

    def delayPanic(self, apsPars):
        '''
        If everything is working correctly, the current values in apsisflat[]
        suffice to carry on.  However, if TelCal is struggling, it will flag
        problems in its report.  At the moment, we have no logic for how to
        help TelCal from this observing script.  ICT-14852 is needed first.
        '''
        msg = 'delayPanic: '
        for bbe in self._basebandEnums:
            bbi = apsPars.indexOf(bbe)
            details = self.bbdreps[bbi].split(':')
            if details[0] == 'flat':
                msg += str(bbe) + '-ok,'
                continue
            elif details[0] == 'set':
                # telcal didn't post a phasing report value
                msg += str(bbe) + '-nc,'
                ## Consider: self.apsisflat[bbi] = False
            else:
                if len(details)>1: dets = str(details[1:])
                else:              dets = 'no details'
                self.logInfo('Delay Fix Issue on %s: %s' % (str(bbe),dets))
                ## Consider:  self.apsisflat[bbi] = False
        self.logInfo(msg)
        self.logInfo("apsisflat:" + str(self.apsisflat) + ' delayPanic')

    def calibration(self):
        '''
        This method runs the growing list of calibrations
        made at the beginning of the observation.
        '''
        self._vlbiTime = APP.getAcsTimestamp()
        self.ano(self, 'v', 'calstart ' + str(self._vlbiTime))
        self.logInfo('Entering pre-VLBI Calibrations')
        #
        self.logInfo('Configuring PICs; this may take a minute')
        APP.APP_PICSetup(self)
        self.logInfo('Delivering schedule to the recorders')
        APP.APP_RecorderSetup(self)
        # SkipAlmaCal is settable for simple tests
        if self._SkipAlmaCal == 1:
            self.logInfo("Skipping group 1 calibrations prior to VLBI Loop")
        else:
            self.logInfo("Perform group 1 calibration targets prior to" +
                "entering the VLBI observing loop")
            self.calibratePriorToVLBILoop()
        #
        # Insert APS Delay Cal targets if they have been created
        # The presumption on fail cases is that TelCal will fall
        # back on TMCDB values.  There may be other options such
        # as retrying with another source....  Note that if we
        # have AppDelayRequired set to 0, self.apscalibs is empty.
        self.logInfo('TelCal Delay Cal %d loops' % self._AppDelayCalLoops)
        if self._AppDelayCalLoops == 0:
            self.logInfo('AppDelayCalLoops is Zero, no Delay Cal')
            return
        for ii in range(self._AppDelayCalLoops):
            failcount = 0
            for rx in self.apscalibs:
                vc = self.apscalibs[rx]
                if vc.calibrated:
                    self.logInfo(vc.personal + ' was already calibrated')
                    continue
                vc.updateTiming()
                self.performVlbiScan(vc, rx=rx)
                if self.delaycalscanpass:
                    self.logInfo(vc.personal + ' is happily calibrated')
                    vc.calibrated = True
                else:
                    failcount += 1
                    ## choose another source, perhaps?
        if self._sessionControl and failcount == 0:
            # i.e. we did all of them
            self.vom.sbe.exStateDict['APSDelayCalDone'] = 'done'
        if 'APSDelayCalDone' in self.vom.sbe.exStateDict:
            self.logInfo('APSDelayCalDone is set in Session to ' +
                self.vom.sbe.exStateDict['APSDelayCalDone'])
        # post bbslope_* values from TelCal into the log
        self.apsdelays.postSlopeValues(self, self._AppDelaySlopeReps)

    def observe(self):
        '''
        This method runs the full set of VLBI observations.
        '''
        self.logInfo('Entering VLBI Observing Loop')
        for vs in self._VLBIScanList:
            # might need to unexpectedly bail early
            if APP.APP_PrematureExitCheck(self, vs): break
            # that file might suggest some changes
            self.ano(self, 'v', 'working %s %s %s ' % (
                vs.SourceName, vs.RecVexTime, str(APP.getAcsTimestamp())))
            APP.APP_VLBIScanUpdates(self, vs)
            self.logInfo('working     %s at %s' % (
                vs.SourceName, vs.RecVexTime))
            self._NextFieldSource = vs.SubscanFieldSource
            # with active phasing we can afford the calibrations
            # with passive phasing we need to leave the TFBs undisturbed
            if vs.ActivePhase:
                self.calibratePriorToScan(vs)
            self.performVlbiScan(vs)
        self.ano(self, 'v', 'postcal ' + str(APP.getAcsTimestamp()))
        self.calibrateAfterScans()
        APP.APP_PICStatus(self)

    def teardown(self):
        '''
        Following the observation, this shuts down, destroying the
        VOM and ending the execution block.
        '''
        self._doneTime = APP.getAcsTimestamp()
        self.ano(self, 'v', 'done ' + str(self._doneTime))
        self.logInfo('Post-observation cleanup')
        if self.vom:
            try:
                APP.APP_RecorderAbort(self)
            except Exception as ex:
                self.logException("Unable to abort recorder schedule", ex)

            if self._sessionControl and hasattr(self.vom, "sbe"):
                self.vom.sbe.endSBExecution()

        if self.array:
            try:
                self.array.endExecution(self._status)
                self.array.setAppSumAntenna(False)
            except Exception as ex:
                self.logException("Unable to endExecution", ex)
        self._exitTime = APP.getAcsTimestamp()
        APP.APP_TimingSummary(self)
        self.ano(self, 'x', 'exit')

    def execute(self):
        '''
        This method executes the complete observation, start to finish.
        self.where is used to provide some limited tracking.
        '''
        try:
            self.where = 'config()'
            self.config()
            self.where = 'prepare()'
            self.prepare()
            self.where = 'calibration()'
            self.calibration()
            self.where = 'observe()'
            self.observe()
        except Exception as ex:
            import traceback
            self.logInfo("%s" % (traceback.format_exc()))
            self.logException(
                "Exception (%s) in %s:" % (type(ex), self.where), ex)
            self.offerWhereHelp()
            self._status = ex
        except KeyboardInterrupt:
            self.logError("^C interrupt...shutting down")
        finally:
            self.teardown()

    def offerWhereHelp(self):
        '''
        This code offers some help following exceptions to
        provide guidance to the Operators or AODs for known issues.
        '''
        if (self.where == 'init'):
            msg = 'Script died during __init__; this should not happen.'
        elif (self.where == 'config()' or
              self.where == 'APP_SelfInit()' or
              self.where == 'APP_SelfSetup()'):
                msg = 'There is likely a problem with ExpertParameter parsing.'
        elif (self.where == 'recycleAttenSettings()' or
              self.where == 'recycleFieldSources()'):
                msg = 'There is an issue with the Session: '
                msg += ' destroy the Array to continue'
        elif (self.where == 'prepare()'):
            msg = 'Script died prior to beginning Execution.'
        elif (self.where == 'APP_FinalRunTimeChecks()'):
            msg = 'Script died checking CAI-63 or the VLBIRecorders.'
        elif (self.where == 'sanity check/overrideVmodeRef'):
            msg = 'VLBI Expert Parameters are misaligned. '
            msg += ' You may need to edit the SB or re-run VEX2VOM.'
        elif (self.where == 'no surviving scans'):
            msg = 'The SB has been launched at a time when there do'
            msg += ' not appear to be scans to observe.  Check the'
            msg += ' observing schedule and try again.'
        elif (self.where == 'SB launched too soon'):
            msg = 'The SB was launched at a time too far from the'
            msg += ' scheduled VLBI scans--this does not make sense.'
        elif (self.where == 'programApsDelaySystem()'):
            msg = 'Unable to get delays from TMCDB.'
        # beginExecution
        elif (self.where == 'APP_ShowHWSetup()'):
            msg = 'There is a problem documenting the LO setttings'
        elif (self.where == 'addSSRModetoVOM()'):
            msg = 'There was an issue adding execution state to VOM. '
            msg += 'There could be some problem satisfying query sources.'
        elif (self.where == 'identifyScienceScans()'):
            msg = 'There was some issue identifying science scans, or '
            msg += 'creating APS calibration targets'
        elif (self.where == 'APP_TelCalConfig()'):
            msg = 'There were issues tuning TelCal parameters.'
        elif (self.where == 'updateAveragingModes()'):
            msg = 'There were problems parsing the AvergingModes'
        elif (self.where == 'updateTelCalForApsDelaySystem()'):
            msg = 'There were problems starting the delay system with TelCal'
        elif (self.where == 'APP_VOMConfig()'):
            msg = 'There were issues setting VOM parameters. '
            msg += ' Are you trying to run on the ACA?'
        elif (self.where == 'calibratePriorToVLBILoop()'):
            msg = 'There is some problem satisfying source queries.'
        elif (self.where == 'calibratePriorToScan()'):
            msg = 'There is some issue calibrating prior to a VLBI scan.'
        elif (self.where == 'performVlbiScan()'):
            msg = 'There is some issue observing a VLBI scan.'
        elif (self.where == 'calibrateAfterScans()'):
            msg = 'There is some issue observing the polarization calibrator.'
        elif (self.where == 'observe()'):
            msg = 'This is most likely an issue with the PICs or Recorders.'
        else:
            msg = 'Create a PRTSPR as this may be a new failure!'
        self.logInfo(msg)


def manualModeExecution():
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option('-a', '--arrayName')
    parser.add_option('-x', '--schedBlockName')
    opts, args = parser.parse_args()
    if opts.arrayName is None or opts.schedBlockName is None:
        parser.print_help()
        sys.exit(1)
    CCL.Global.setArrayName(opts.arrayName)
    array = CCL.Global.getArray(opts.arrayName)
    CCL.Global.loadSB(opts.schedBlockName)
    vlbi = StandardVLBI(scriptName="StandardVLBI.py", scriptArgs=" ".join(args))


if __name__ == "__main__":
    # Manual array execution
    manualModeExecution()
else:
    # Interactive array execution
    # __name__ == builtins     (Py3)
    # __name__ == __builtin__  (Py2)
    # "ObservingModeSimulator" in __name__
    vlbi = StandardVLBI()
#
# eof
#
