#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalSolarFastScan.py 247933 2017-08-08 15:57:43Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global math
import math

global Control
import Control

global CCL
import CCL.APDMSchedBlock

global ControlExceptionsImpl
import ControlExceptionsImpl

global PyDataModelEnumeration
import PyDataModelEnumeration

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase
import Observation.FillPattern
import Observation.FastScanPattern
import Observation.CalibratorSource


class ObsCalSolarFastScan(Observation.ObsCalBase.ObsCalBase):

    pointingExcursion = None
    # These are on-source levels
    ifLevel = -26.0
    bbLevel = 2.0
    mapAttenuatorSettingName = "Map attenuator setting"
    mapSubscanSequenceSpec = None
    mapMixerMode = Control.NORMAL
    _srcPointFocus = None

    options = [
        Observation.ObsCalBase.scriptOption("pointingSubscanDuration", float, 3.072),
        Observation.ObsCalBase.scriptOption("focusSubscanDuration", float, 2.0),
        Observation.ObsCalBase.scriptOption("atmSubscanDuration", float, 4.0),
        Observation.ObsCalBase.scriptOption("numFocusPositions", int, 5),
        Observation.ObsCalBase.scriptOption("focusOneWay", bool, False),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.001),
        # map parameters
        Observation.ObsCalBase.scriptOption("mapPatternType", str, "CENTERED"),
        Observation.ObsCalBase.scriptOption("mapPatternSubtype", str, "AUTO"),
        Observation.ObsCalBase.scriptOption("mapWidth", float, 2400.0),
        Observation.ObsCalBase.scriptOption("mapHeight", float, 2400.0),
        # Negative value means to use a band-dependent default value
        Observation.ObsCalBase.scriptOption("mapSpacing", float, -1.0),
        Observation.ObsCalBase.scriptOption("mapOrientation", float, 0.0),
        Observation.ObsCalBase.scriptOption("mapMaxFreq", float, 1.0),
        Observation.ObsCalBase.scriptOption("mapDoRef", bool, True),
        Observation.ObsCalBase.scriptOption("mapRefSubscanDuration", float, 10.0),
        # Negative value means to use a band-dependent default value
        Observation.ObsCalBase.scriptOption("mapDuration", float, -1.0),
        # As we sometimes observe the Moon or Sun we are best to default to a long way away from the source!
        Observation.ObsCalBase.scriptOption("atmAzOffsetArcsec", float, 3600.0),
        Observation.ObsCalBase.scriptOption("atmDoZero", bool, True),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "2 deg"),
        # Negative value means to use a default value
        Observation.ObsCalBase.scriptOption("band", int, -1),
        Observation.ObsCalBase.scriptOption("source", str, "Sun"),
        Observation.ObsCalBase.scriptOption("activeSun", bool, False),
        Observation.ObsCalBase.scriptOption("mapLongOff", float, 0.),
        Observation.ObsCalBase.scriptOption("mapLatOff", float, 0.),
        Observation.ObsCalBase.scriptOption("noPntFocus", bool, False),
        Observation.ObsCalBase.scriptOption("atmCycleTime", float, 720.),
        Observation.ObsCalBase.scriptOption("removeCorrSetup", bool, False),
        Observation.ObsCalBase.scriptOption("addSQLD", bool, False),
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self, singleDish=True)

        self._useAPDM = False
        if self._sb:
            # SB execution
            sb = Observation.SchedulingBlock.SchedulingBlock()
            self._ssrSB = sb

            target = self._ssrSB.getRepresentativeTarget()
            fp = target.getSource().getFieldPattern()
            if isinstance(fp, CCL.APDMSchedBlock.RectanglePattern):
                # As of now, OT puts RectanglePattern to the target source
                # for the Solar full-disk mapping
                self._useAPDM = False
            else:
                self._useAPDM = True

            msg = "Running %s scheduling block." % (sb._controlSB.name)
        else:
            # Manual mode execution
            self._ssrSB = None
            msg = "Running manual mode command."

        if self._useAPDM:
            msg += " Will use APDM info to configure scan pattern"
        else:
            msg += " Will use expert parameters to configure scan pattern"
        self.logInfo(msg)

    def parseOptions(self):
        # In ICT-6959 we chose to use the SB representative band instead of expert param
        self.band = int(3)
        if self._sb is not None:
            repBandStr = str(self._sb.SchedulingConstraints.representativeReceiverBand)
            self.band = int(repBandStr.split('_')[-1])
            self.logInfo("SB Representative band: %d ('%s')" % (self.band, repBandStr))
        # expert band parameter or command-line option takes precedence if used
        if self.args.band > 0:
            self.logInfo("User-specified band: %d" % self.args.band)
            self.band = self.args.band
        # Use some canned defaults for solar/moon maps in each band.
        # The duration should be computed automatically from the pattern spec.
        self.mapSpacing = 20.0
        # self.mapDuration = 290.0
        if self.band == 5:
            self.mapSpacing = 12.0
        if self.band == 6:
            self.mapSpacing = 10.0
            # self.mapDuration = 570.0
        if self.band == 7:
            self.mapSpacing = 8.0
            # self.mapDuration = 710.0
        if self.band == 9:
            self.mapSpacing = 4.0
            # self.mapDuration = 1406.0
        self.pointingSubscanDuration = self.args.pointingSubscanDuration
        self.focusSubscanDuration    = self.args.focusSubscanDuration
        self.atmSubscanDuration      = self.args.atmSubscanDuration
        self.numFocusPositions       = self.args.numFocusPositions
        self.focusOneWay             = self.args.focusOneWay
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        # map parameters
        self.mapPatternType          = self.args.mapPatternType
        self.mapPatternSubtype       = self.args.mapPatternSubtype
        self.mapWidth                = self.args.mapWidth
        self.mapHeight               = self.args.mapHeight
        if self.args.mapSpacing > 0.0:
            self.logInfo("User-specified map spacing: %.3f arcsec" % self.args.mapSpacing)
            self.mapSpacing          = self.args.mapSpacing
        self.mapOrientation          = self.args.mapOrientation
        self.mapMaxFreq              = self.args.mapMaxFreq
        self.mapDoRef                = self.args.mapDoRef
        self.mapRefSubscanDuration   = self.args.mapRefSubscanDuration
        # if self.args.mapDuration > 0.0:
        #     self.logInfo("User-specified map duration: %.3f seconds" % self.args.mapDuration)
        #     self.mapDuration         = self.args.mapDuration
        self.atmAzOffsetArcsec       = self.args.atmAzOffsetArcsec
        self.atmDoZero               = self.args.atmDoZero
        self.elLimit                 = self.args.ElLimit
        self.sourceName              = self.args.source
        self.activeSun               = self.args.activeSun
        self.noPntFocus              = self.args.noPntFocus

        self.logInfo("Selected band: %d, map spacing: %.3f arcsec" % (self.band, self.mapSpacing))

        if self.band == 7 and self.activeSun:
            msg = "Deactivate activeSun parameter for band 7 observation"
            self.activeSun = False

        if self.activeSun:
            self.mapMixerMode = Control.MD2
        self.atmAzOffsetRad = math.radians(self.atmAzOffsetArcsec / 3600.0)

        # Added 10 seconds margin to take into account that mounts will not
        # be able to catch up for the first few seconds. The margin should be
        # revised later.
        self.timeToCatchUpTrackingPattern = 10.

        # (MMEX onlY) Should be set by setMapSource()
        self.mapSrc = None
        # (MMEX only) These two should be set by generateTunings()
        self._mapSpectralSpec = None
        self._pointFocusSpectralSpec = None
        # (MMEX only) These two should be set by configureScanPattern()
        self._mapOffsetSpec = None
        self._mapDuration = None

    def generateTunings(self):
        frequency = Observation.SSRTuning.bandFreqs_solar[self.band]
        # point/focus tuning will typically have longer sample time than fast scan map
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.band,
                # TODO: should be careful here for bands where solar tuning
                # puts a sideband in an atmospheric line
                frequency = frequency,
                intent = "total_power",
                tpSampleTime = 0.016)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.band

        if self._useAPDM:
            for target in self._ssrSB.getScienceTargetList():
                sp = target.getSpectralSpec()
                if self.args.addSQLD:
                    if not hasattr(sp,'SquareLawSetup') or \
                        sp.SquareLawSetup is None:
                        sp.SquareLawSetup = CCL.APDMSchedBlock.SquareLawSetup()
                        sp.SquareLawSetup.integrationDuration.set(self.tpIntegrationDuration)
                if self.args.removeCorrSetup:
                    sp.ACACorrelatorConfiguration = None
                target.setSpectralSpec(sp)
        else:
            mapSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                                        band = self.band,
                                        frequency = frequency,
                                        intent = "total_power",
                                        tpSampleTime = self.tpIntegrationDuration)
            mapSpectralSpec.name = "Band %d mapping" % self.band

            self._mapSpectralSpec = mapSpectralSpec
    def setMapSource(self):
        if not self._useAPDM:
            self.mapSrc = self.sourceHelper.getSource(self.sourceName)

    def setTelCalParams(self):
        from Observation.Global  import simulatedArray
        self.logInfo("Setting TelCal parameters pointingFitWidth=False, simpleGaussianFit=False")
        if simulatedArray():
            return
        tcParameters = self.getTelCalParams()
        tcParameters.setCalibParameter('pointingFitWidth', False)
        tcParameters.setCalibParameter('simpleGaussianFit',False)



    # This should probably move to PointingCalTarget
    def setPointingExcursion(self):
        clight = 299792458.0
        antennaDiameter = 12.0
        # doing this here where we can't account for pointing subarrays is a bit dumb
        for ant in self._array.antennas():
                if ant[0] == 'C':
                        antennaDiameter = 7.0
                        self.logInfo("Array contains at least one CM antenna, so using D=7m for beam sizes.")
                        break
        obsFreq = self._pointFocusSpectralSpec.getMeanFrequency()
        sourceDiameter = 0.0
        try:
            source = Observation.CalibratorSource.SSOCalibratorSource(self._srcPointFocus.sourceName)
            sourceDiameter = math.radians(source.getAngularDiameter() / 3600.0)
        except Exception as ex:
            self.logException("Exception thrown when computing source diameter, assuming zero diameter.", ex)
        self.logInfo("Assumed source diameter: %.2f arcsec" % (3600.0 * math.degrees(sourceDiameter)))
        beam = 1.22 * (clight / obsFreq) / antennaDiameter
        excursion = math.sqrt(4.0*beam*beam + sourceDiameter*sourceDiameter)
        self.logInfo("Assumed antenna diamter: %.1f m" % (antennaDiameter))
        self.logInfo("Assumed beam FWHM:       %.2f arcsec" % (3600.0 * math.degrees(beam)))
        self.logInfo("Chosen excursion:        %.2f arcsec" % (3600.0 * math.degrees(excursion)))
        self.pointingExcursion = excursion


    def doPointing(self):
        if self._srcPointFocus is None:
            return
        if self.pointingExcursion is None:
            self.setPointingExcursion()
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('TOTAL_POWER')
            pointingCal.setDelayCalReduction(False)
            pointingCal.setPointingMethod('cross')
            pointingCal.setExcursion(self.pointingExcursion)
            pointingCal.setWVRCalReduction(False)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex


    def doFocus(self):
        if self._srcPointFocus is None:
            return
        ref = CCL.APDMSchedBlock.Reference()
        ref.setCoordinates('-600 arcsec',0,u'horizon')
        ref.cycleTime.set(1.0)
        ref.subScanDuration.set(self.focusSubscanDuration)
        ref.integrationTime.set(self.focusSubscanDuration)
        self._srcPointFocus.Reference = [ref]
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            focusCal.setSubscanDuration(self.focusSubscanDuration)
            focusCal.setIntegrationTime(3.0 * self.focusSubscanDuration)
            focusCal.setDataOrigin('TOTAL_POWER')
            focusCal.setNumPositions(self.numFocusPositions)
            focusCal.setOneWayScan(self.focusOneWay)
            focusCal.setWVRCalReduction(False)
            self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
            # ICT-9503: avoid reference getting used by pointing or any other scans by accident.
            self._srcPointFocus.Reference = []
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                             (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No focus results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex


    def doAtmCal(self, src=None, spectralSpec=None):
        if src is None:
            src = self.mapSrc
        if spectralSpec is None:
            spectralSpec = self._mapSpectralSpec
        try:
            atm = Observation.AtmCalTarget.AtmCalTarget(src, spectralSpec, doHotLoad=True)
            isObs = atm.isObservable(observingMode=self._obsmode)
            if not isObs:
                msg = "'%s' is not observable" % (atm)
                raise Exception(msg)
            atm.setOnlineProcessing(True)
            atm.setDoZero(self.atmDoZero)
            atm.setSubscanDuration(self.atmSubscanDuration)
            atm.setIntegrationTime(1.5)
            atm.setMixerMode(self.mapMixerMode)
            if spectralSpec.hasCorrelatorConfiguration():
                    atm.setDataOrigin('FULL_RESOLUTION_AUTO')
                    atm.setWVRCalReduction(True)
            else:
                    atm.setDataOrigin('TOTAL_POWER')
                    atm.setWVRCalReduction(False)
            atm.setApplyWVR(False)
            # Currently we need to set a reference source to use an offset.
            atm._referenceSource = atm._source
            atm._referenceOffset = CCL.SourceOffset.stroke(self.atmAzOffsetRad,
                                                           0, 0, 0,
                                                           Control.HORIZON)
            self.logInfo('Executing AtmCal on ' + src.sourceName + '...')
            atm.execute(self._obsmode)
            self.logInfo('Completed AtmCal on ' + src.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing AtmCal on source %s" % src.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def prepareMapSubscanSequenceSpec(self, src, spectralSpec, mapOffsetSpec, mapDuration, referenceOffset=None):
        # Prepare subscan sequence specification
        ssl = CCL.SubscanList.SubscanList()
        subscanDuration = 0.048 * (1 + int(mapDuration / 0.048))
        ssl.addSubscan(src, spectralSpec, subscanDuration,
                        SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.ON_SOURCE,
                        CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                        PointingOffsetSpec    = mapOffsetSpec,
                        attenuatorSettingName = self.mapAttenuatorSettingName,
                        MixerMode             = self.mapMixerMode)

        if self.mapDoRef:
            if referenceOffset is None:
                referenceOffset = CCL.SourceOffset.stroke(self.atmAzOffsetRad, 0, 0, 0, Control.HORIZON)
            self.logInfo("referenceOffset = %s" % str(referenceOffset))
            subscanDuration = 0.048 * (1 + int(self.mapRefSubscanDuration / 0.048))
            ssl.addSubscan(src, spectralSpec, subscanDuration,
                           SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.OFF_SOURCE,
                           CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                           PointingOffsetSpec    = referenceOffset,
                           attenuatorSettingName = self.mapAttenuatorSettingName,
                           MixerMode             = self.mapMixerMode)
        subscanSpec = ssl.getSubscanSequenceSpecification()
        self.logInfo('subscanSpec.subscans = %s' % str(subscanSpec.subscans))
        self.mapSubscanSequenceSpec = subscanSpec

    def enableFocusModels(self, enable, continueOnError=False, noExcept=False):
        enableStr = "Enabling" if enable else "Disabling"
        self.logInfo(enableStr + " focus models...")
        try:
            amc = self._obsmode.getArrayMountController()
            amc.enableFocusModel(enable)
            isEnabled = amc.isFocusModelEnabled()
            # consider raising an exception if isEnabled != enable
            self.logInfo("ArrayMountController reports that focus model enable is %s" % str(isEnabled))
        except BaseException as ex:
                print(ex)
                msg = "Error %s focus models in ArrayMountController" % (enableStr)
                self.logError(msg)
                if not noExcept:
                    raise ex

    def executeMapScan(self, src):
            self.logInfo("Parking ACD so we optimise attenuators on-sky...")
            try:
                self._obsmode.setCalibrationDevice(PyDataModelEnumeration.PyCalibrationDevice.NONE)
            except BaseException as ex:
                self.logException('Exception thrown by obsmode.setCalibrationDevice(), considering this fatal!', ex)
                raise ex
            # Just before execution turn off the focus model when pointed at the source
            self.logInfo("Slewing to source with obsmode.setPointingDirection() to set focus before disabling focus models...")
            try:
                self._obsmode.setPointingDirection(src)
            except BaseException as ex:
                self.logException('Exception thrown by obsmode.setPointingDirection(), considering this fatal!', ex)
                raise ex
            try:
                self.enableFocusModels(enable=False, continueOnError=False, noExcept=False)
            except BaseException as ex:
                self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
                raise ex
            # the obvserving mode seems to set the levels after starting the motion pattern, so we have to do it ourselves and save the result. Ugh.
            self.logInfo("Optimising attenuators and saving to setting named '%s'" % self.mapAttenuatorSettingName)
            try:
                loMode = self._obsmode.getLOObservingMode()
                loMode.optimizeSignalLevels(ifTargetLevel = self.ifLevel, bbTargetLevel = self.bbLevel, settingName = self.mapAttenuatorSettingName)
            except BaseException as ex:
                self.logException('Exception thrown by optimizeSignalLevels() or getting reference to LOObservingMode, considering this fatal!', ex)
                raise ex
            # Sort out correlator calibration IDs
            if self.mapSubscanSequenceSpec.spectra[0].hasCorrelatorConfiguration():
                self.logInfo('Performing correlator calibration via getCalibrationIds() with validity interval %d seconds'
                                    % self._obsmode.getCorrelatorCalibrationValidityInterval())
                try:
                    self._obsmode.getCalibrationIds(self.mapSubscanSequenceSpec)
                except Exception as ex:
                    self.logException('Exception thrown by obsmode.getCalibrationIds(), considering this fatal!', ex)
                    raise ex
            else:
                self.logInfo('No correlator configuration, so just setting calibration IDs to -1 to save effort calling getCalibrationIds()')
                for s in self.mapSubscanSequenceSpec.subscans:
                    s.calibrationId = -1
            # Execute the sequence
            self.logInfo("Now executing the subscan sequence...")
            try:
                self._obsmode.doSubscanSequence(self.mapSubscanSequenceSpec)
            except Exception as ex:
                self.logException('Exception thrown by obsmode.doSubscanSequence(), considering this fatal!', ex)
                raise ex
            self.logInfo('Subscan sequence finished')
            # Re-enable focus models, allowing an exception if any antennas fail
            try:
                self.enableFocusModels(enable=True, continueOnError=True, noExcept=False)
            except BaseException as ex:
                self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
                raise ex

    def doMap(self, src=None, spectralSpec=None, mapOffsetSpec=None, mapDuration=None, referenceOffset=None):
        if src is None:
            src = self.mapSrc
        if spectralSpec is None:
            spectralSpec = self._mapSpectralSpec
        if mapOffsetSpec is None:
            mapOffsetSpec = self._mapOffsetSpec
        if mapDuration is None:
            mapDuration = self._mapDuration

        self.prepareMapSubscanSequenceSpec(src, spectralSpec,
                                           mapOffsetSpec, mapDuration,
                                           referenceOffset=referenceOffset)

        scanIntent = CCL.ScanIntent.ScanIntent(PyDataModelEnumeration.PyScanIntent.OBSERVE_TARGET)
        self.logInfo('Now beginning the scan')
        self._obsmode.beginScan(scanIntent)
        try:
            self.executeMapScan(src)
        except BaseException as ex:
            self.logError("Error executing map scan, cleaning up and exiting")
            self.enableFocusModels(enable=True, continueOnError=True, noExcept=True)
            self._obsmode.endScan()
            self.closeExecution(ex)
            raise ex
        self._obsmode.endScan()
        self.logInfo("Scan ended")

    def observeTarget(self, target):
        import time
        from Observation.PointingCalTarget import PointingCalTarget
        from Observation.Global  import simulatedArray

        src = target.getSource()

        mapSpectralSpec = target.getSpectralSpec()

        # Do Map
        if len(src.Reference) > 0:
            refSource, refOffset = target.parseReferencePosition(0)
        else:
            refOffset = None

        targetIntegrationTime = target.getIntegrationTime()
        mapDuration = target.singleMapDuration

        # How many maps we can make within the cycle time
        nMapCycle = int(float(self.args.atmCycleTime) / mapDuration)
        if (nMapCycle + 1) * mapDuration < self.args.atmCycleTime * 1.1:
            nMapCycle += 1

        self.logInfo("targetIntegrationTime=%s" % (targetIntegrationTime))
        self.logInfo("mapDuration=%s" % (mapDuration))
        self.logInfo("nMapCycle=%s" % (nMapCycle))

        while 1:
            # cIntegTime = target.getCurrentIntegrationTime()
            remain = target.getIntegrationTime() - target.getCurrentIntegrationTime()
            if remain <= 0:
                break

            self.doAtmCal(src, target.getSpectralSpec())

            # How many maps required to achieve the target integration time
            nMapRemain = int(float(remain) / mapDuration)
            if nMapRemain * mapDuration < remain:
                nMapRemain += 1

            nMap = min(nMapRemain, nMapCycle)
            if nMap <= 0:
                nMap = 1

            self.logInfo("Will do %d map(s) (%.1f seconds in total)" % (nMap, mapDuration * nMap))
            # for iMap in range(nMap):
            duration = mapDuration * nMap
            msg = "Integration time = [%.1f/%.1f]" % (target.getCurrentIntegrationTime(), targetIntegrationTime)
            self.logInfo(msg)
            self.logInfo("remain=%d" % (remain))
            self.doMap(src, mapSpectralSpec, target.mapOffsetSpec, duration,
                       referenceOffset=refOffset)
            target.setCurrentIntegrationTime(target.getCurrentIntegrationTime() + duration)

    def observeTargets(self):
        try:
            if not self._useAPDM:
                self.logInfo("Executing AtmCal...")
                self.doAtmCal()
                self.logInfo("Executing Map...")
                self.doMap()
                self.logInfo("Executing AtmCal...")
                self.doAtmCal()
                return

            groupList = self._ssrSB.getGroupList()

            lastTarget = None
            for group in groupList[1:]:
                msg = "[Group%s]" % (group.groupIndex)
                self.logInfo(msg)
                for target in group.getScienceTargetList():
                    target.setCurrentIntegrationTime(0.)
                    self.observeTarget(target)
                    lastTarget = target

            if lastTarget:
                self.doAtmCal(lastTarget.getSource(), lastTarget.getSpectralSpec())

        except BaseException as ex:
            import traceback
            self.logError(str(traceback.format_exc()))
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def configureScanPatternUsingExpertParameters(self):
        import Observation.FillPattern
        self.logInfo("Try to generate map offset spec using expert parameters")
        patternType = Observation.FillPattern.FillPatternType.getType(self.mapPatternType)
        patternSubType = Observation.FillPattern.FillPatternSubtype.getSubtype(self.mapPatternSubtype)
        fp = Observation.FillPattern.FillPattern('%.1f arcsec' % self.mapWidth,
                                                 '%.1f arcsec' % self.mapHeight,
                                                 '%.1f arcsec' % self.mapSpacing,
                                                 '%.3f deg' % self.mapOrientation,
                                                 patternType,
                                                 patternSubType)

        lonOff = math.radians(self.args.mapLongOff / 3600.)
        latOff = math.radians(self.args.mapLatOff / 3600.)
        #   self.logInfo("Fill Pattern Type = %s" % (fillPatternType))
        #   self.logInfo("Fill Pattern Sub-Type = %s" % (fillPatternSubType))
        mapOffsetSpec, mapDuration = self.generateMapOffsetSpec(fp, lonOff, latOff)

        if self.args.mapDuration > 0.0:
            self.logInfo("User-specified map duration: %.3f seconds" % self.args.mapDuration)
            mapDuration = self.args.mapDuration

        self._mapOffsetSpec = mapOffsetSpec
        self._mapDuration = mapDuration

    def generateMapOffsetSpec(self, fp, lonOff, latOff):
        fsp = Observation.FastScanPattern.FastScanPattern(verbose=True)

        self.logInfo("Overridden limitFrequencyMax %.3f -> %.3f" % (fsp.limitFrequencyMax, self.mapMaxFreq))
        fsp.limitFrequencyMax = self.mapMaxFreq

        spec, mapDuration = fsp.generatePatternSpec(fp, returnDuration=True)
        mapDuration = mapDuration + self.timeToCatchUpTrackingPattern

        # frame = fp.scanningCoordinateSystem
        self.logInfo("scanningCoordinateSystem = '%s'" % (fp.scanningCoordinateSystem))

        # # FIXME: fp.scanningCoordinateSystem is not accepted by Control
        # # fp.scanningCoordinateSystem is of 'str' type, but doSubscanSequence does not accept it
        # if str(fp.scanningCoordinateSystem.lower()) == 'horizon':
        #     frame = Control.HORIZON
        # elif str(fp.scanningCoordinateSystem.lower()) == 'icrs':
        #     frame = Control.EQUATORIAL
        # else:
        #     assert(False)

        # Always scan in the horizon frame
        frame = Control.HORIZON
        self.logInfo("Map pattern spec: " + str(spec._PatternSpec__idlSpec))

        mapOffsetSpec = Control.SourceOffset(
            longitudeOffset=lonOff,
            latitudeOffset=latOff,
            longitudeVelocity=0.0,
            latitudeVelocity=0.0,
            pattern=spec._PatternSpec__idlSpec,
            frame=frame
        )

        self.logInfo("Map offset spec: " + str(mapOffsetSpec))
        self.logInfo("Single map duration: %.2f [seconds]" % (mapDuration))

        return mapOffsetSpec, mapDuration


    def configureScanPatternUsingAPDMParameters(self):
        self.logInfo("Try to generate map offset spec using APDM parameters")
        for group in self._ssrSB.getGroupList():
            for target in group.getScienceTargetList():
                src = target.getSource()
                fp = src.getFieldPattern()
                self.logInfo(fp.toDOM().toprettyxml())
                if fp is None:
                    msg = "'%s' does not have FillPattern" % (target)
                    raise Exception(msg)
                self.logInfo("Field Patern: %s" % (fp.toDOM().toxml()))
                assert(fp is not None)

                lonOff = fp.patternCenterCoordinates.longitude.get()
                latOff = fp.patternCenterCoordinates.longitude.get()

                mapOffsetSpec, mapDuration = self.generateMapOffsetSpec(fp, lonOff, latOff)
                target.mapOffsetSpec = mapOffsetSpec
                target.singleMapDuration = mapDuration

    def configureScanPattern(self):
        if self._useAPDM:
            self.configureScanPatternUsingAPDMParameters()
        else:
            self.configureScanPatternUsingExpertParameters()


obs = ObsCalSolarFastScan()
obs.parseOptions()
#obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.setMapSource()
    # TODO: pass source name parameter sunMoon here so can look for a close source
    if not obs.noPntFocus:
        obs.findPointFocusSource(minEl=25.0, maxEl=80.0, singleDish=True)
    obs.configureScanPattern()
except BaseException as ex:
    import traceback
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.logError(str(traceback.format_exc()))
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.setTelCalParams()
obs.logInfo("Executing first pointing...")
obs.doPointing()
obs.logInfo("Executing second pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing focus...")
obs.doFocus()
obs.logInfo("Executing last pointing -- make sure results are good!...")
obs.doPointing()
obs.observeTargets()
obs.closeExecution()

